<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use GuzzleHttp\Exception\BadResponseException;
use Rowlinson\Api\Exceptions\ApiException;
use Rowlinson\Api\Requests\Basket\AddItemRequest;
use Rowlinson\Api\Requests\Basket\ConfirmRequest;
use Rowlinson\Api\Requests\Basket\OrderTypeRequest;
use Rowlinson\Api\Requests\Basket\UpdateItemRequest;
use Rowlinson\Api\Requests\Basket\ValidateRequest;
use Rowlinson\Api\Responses\Basket\AddFromOrderResponse;
use Rowlinson\Api\Responses\Basket\BasketResponse;
use Rowlinson\Api\Responses\Basket\DeleteBasketResponse;
use Rowlinson\Api\Responses\Basket\DeleteItemResponse;
use Rowlinson\Api\Responses\Basket\UpdateItemResponse;
use Rowlinson\Api\Responses\Basket\ValidateResponse;
use Symfony\Component\HttpFoundation\Response;

class BasketClient extends AbstractClient
{
    /**
     * Add a product variant to the currently logged in user's basket
     */
    public function addToBasket(string $sku, int $quantity): void
    {
        $this->post('basket/add_item', new AddItemRequest($sku, $quantity));
    }

    /**
     * Add many product variants to the currently logged in user's basket
     *
     * @param array<string,int> $items Key is the SKU, value is the quantity
     */
    public function addManyToBasket(array $items): BasketResponse
    {
        $response = $this->post(
            'basket/add_items',
            array_map(fn(int $q, string $s) => new AddItemRequest($s, $q), $items, array_keys($items))
        );

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            BasketResponse::class,
            'json'
        );
    }

    /**
     * Get the currently logged in user's basket
     */
    public function getBasket(): BasketResponse
    {
        $response = $this->get('basket');

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            BasketResponse::class,
            'json'
        );
    }

    /**
     * Remove the basket entirely
     */
    public function removeBasket(): DeleteBasketResponse
    {
        $response = $this->delete('basket');

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            DeleteBasketResponse::class,
            'json'
        );
    }

    /**
     * Updates a single basket item
     */
    public function updateBasketItem(string $variantSku, int $quantity): UpdateItemResponse
    {
        $response = $this->put('basket/update_item', new UpdateItemRequest($variantSku, $quantity));

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            UpdateItemResponse::class,
            'json'
        );
    }

    /**
     * Removes a single item from the users basket
     */
    public function removeBasketItem(string $sku): DeleteItemResponse
    {
        $response = $this->delete('basket/remove_item/' . $sku);

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            DeleteItemResponse::class,
            'json'
        );
    }

    /**
     * Populate the basket with available items from the supplied order
     */
    public function addBasketItemsFromOrder(string $orderNumber): AddFromOrderResponse
    {
        $response = $this->post('basket/add_order/' . $orderNumber);

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            AddFromOrderResponse::class,
            'json'
        );
    }

    /**
     * Validate a hypothetical basket of items against an optional organisation (admin only)
     *
     * @param array<array<string,string|int>> $items An array of arrays, [ 'sku' => 'ABC/123', 'quantity' => 12 ]
     * @throws ApiException
     */
    public function validate(array $items, ?string $organisation = null): ValidateResponse
    {
        $response = null;

        try {
            $response = $this->post('basket/validate', new ValidateRequest($items, $organisation));
        } catch (ApiException $e) {
            if ($e->getCode() !== Response::HTTP_BAD_REQUEST || !($e->getPrevious() instanceof BadResponseException)) {
                throw $e;
            }

            $response = $e->getPrevious()->getResponse();
        }

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            ValidateResponse::class,
            'json'
        );
    }

    /**
     * Using the current basket for the user, place an order with the supplied details
     *
     * @param array<string,string>|null $newDeliveryLocation
     */
    public function confirm(
        ?string $deliveryLocationId,
        ?string $customerReference,
        ?string $deliveryInstructions,
        ?array $newDeliveryLocation,
        ?string $orderType,
        ?string $deduplicationKey
    ): void {
        $headers = [];
        if ($deduplicationKey !== null) {
            $headers['Deduplication-Key'] = $deduplicationKey;
        }

        $this->post(
            'basket/confirm',
            new ConfirmRequest(
                $deliveryLocationId,
                $customerReference,
                $deliveryInstructions,
                $newDeliveryLocation,
                $orderType
            ),
            $headers
        );
    }

    /**
     * @param int $optionId Delivery option ID e.g. from DeliveryLocationClient::getDeliveryOptions()
     */
    public function setDeliveryOption(int $optionId): BasketResponse
    {
        $response = $this->put('basket/delivery/' . $optionId);

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            BasketResponse::class,
            'json'
        );
    }

    public function setOrderType(?string $orderType): BasketResponse
    {
        $response = $this->put('basket/order_type', new OrderTypeRequest($orderType));

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            BasketResponse::class,
            'json'
        );
    }
}
