<?php

declare(strict_types=1);

namespace Rowlinson\Api\Responses;

use GuzzleHttp\Psr7\Response;

/**
 * Adds helper methods to standard Guzzle response class
 *
 * @package Rowlinson\Api\Responses
 */
class ApiResponse extends Response
{
    /**
     * Create a new API response from a standard Guzzle Response object
     */
    public static function createFromGuzzleResponse(Response $response): self
    {
        return new self(
            $response->getStatusCode(),
            $response->getHeaders(),
            $response->getBody(),
            $response->getProtocolVersion(),
            $response->getReasonPhrase()
        );
    }

    /**
     * Get the total number of items in a paginated set
     *
     * E.g. if the result set was paginated at 30 items, this may return 92. Defaults to 1 if the result isn't paginated
     */
    public function getPaginationTotalItems(): int
    {
        $headers = $this->getHeader('X-Pagination-TotalItems');
        if (empty($headers)) {
            return 1;
        }

        return intval(reset($headers));
    }

    /**
     * Get the number of items per page for a paginated set
     */
    public function getPaginationPerPage(): int
    {
        $headers = $this->getHeader('X-Pagination-PerPage');
        if (empty($headers)) {
            return 50;
        }

        return intval(reset($headers));
    }

    /**
     * Get the property that the collection is sorted by
     */
    public function getSortedBy(): string
    {
        $headers = $this->getHeader('X-SortedBy');
        if (empty($headers)) {
            return 'updated';
        }

        return strval(reset($headers));
    }

    /**
     * Get the direction that the collection is sorted by
     */
    public function getSortDirection(): string
    {
        $headers = $this->getHeader('X-SortDir');
        if (empty($headers)) {
            return 'DESC';
        }

        return strval(reset($headers));
    }
}
