<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use Rowlinson\Api\Collections\PaginatedCollection;
use Rowlinson\Api\Requests\DeliveryLocation\CreateUpdateRequest;
use Rowlinson\Api\Responses\DeliveryLocation\DeliveryLocationResponse;
use Rowlinson\Api\Responses\DeliveryLocation\DeliveryOptionResponse;

class DeliveryLocationClient extends AbstractClient
{
    /**
     * Request the creation of a new delivery address
     */
    public function create(
        string $name,
        string $addressLine1,
        ?string $addressLine2,
        string $town,
        ?string $county,
        string $country,
        string $postcode
    ): void {
        $this->post('delivery_locations', new CreateUpdateRequest(
            $name,
            $addressLine1,
            $addressLine2,
            $town,
            $county,
            $country,
            $postcode
        ));
    }

    /**
     * Request the creation of a new delivery address
     */
    public function edit(
        string $id,
        string $name,
        string $addressLine1,
        ?string $addressLine2,
        string $town,
        ?string $county,
        string $country,
        string $postcode
    ): void {
        $this->put('delivery_locations/' . $id, new CreateUpdateRequest(
            $name,
            $addressLine1,
            $addressLine2,
            $town,
            $county,
            $country,
            $postcode
        ));
    }

    /**
     * Remove a delivery location
     */
    public function remove(string $id): void
    {
        $this->delete('delivery_locations/' . $id);
    }

    /**
     * Get the delivery locations for the current user
     *
     * @return PaginatedCollection<DeliveryLocationResponse>
     */
    public function getDeliveryLocations(
        ?string $organisation = null,
        int $page = 1,
        int $perPage = 10
    ): PaginatedCollection {
        $response = $this->get('delivery_locations?' . http_build_query(array_filter([
            'page' => $page,
            'per_page' => $perPage,
            'organisation' => $organisation,
        ])));

        $deliveryResponses = $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . DeliveryLocationResponse::class . '>',
            'json'
        );

        return new PaginatedCollection($deliveryResponses, $page, $perPage, $response->getPaginationTotalItems());
    }

    /**
     * @return array<DeliveryOptionResponse>
     */
    public function getDeliveryOptions(?string $orderType = null): array
    {
        $queryString = '';
        if ($orderType !== null) {
            $queryString = '?' . http_build_query([ 'order_type' => $orderType ]);
        }

        $response = $this->get('delivery_options' . $queryString);

        /** @var array<DeliveryOptionResponse> */
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . DeliveryOptionResponse::class . '>',
            'json'
        );
    }
}
