<?php

declare(strict_types=1);

namespace Rowlinson\Api\Collections;

/**
 * @template T
 * @extends PaginatedCollection<T>
 */
final class SortedPaginatedCollection extends PaginatedCollection
{
    private string $sortedBy;

    private string $sortDir;

    /**
     * @param array<int|string,T> $items
     * @param int $page The page this collection is of the total
     * @param int $perPage How many items per page was requested
     * @param int $total The total number of items if this collection weren't paginated
     */
    public function __construct(
        string $sortedBy,
        string $sortDir,
        array $items = [],
        int $page = 1,
        int $perPage = 10,
        int $total = 1
    ) {
        parent::__construct($items, $page, $perPage, $total);

        $this->sortedBy = $sortedBy;
        $this->sortDir = $sortDir;
    }

    public function getSortedBy(): string
    {
        return $this->sortedBy;
    }

    public function getSortDirection(): string
    {
        return $this->sortDir;
    }


    /**
     * @template T2 The mapped collection's type
     * @param callable(T): T2 $callback
     * @return SortedPaginatedCollection<T2>
     */
    public function map(callable $callback): self
    {
        $keys = array_keys($this->items);
        $items = array_map($callback, $this->items, $keys);

        return new static(
            $this->sortedBy,
            $this->sortDir,
            array_combine($keys, $items),
            $this->page,
            $this->perPage,
            $this->total
        );
    }
}
