<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\ServerException;
use JMS\Serializer\SerializerInterface;
use Rowlinson\Api\Credentials\Credentials;
use Rowlinson\Api\Exceptions\ConflictException;
use Rowlinson\Api\Exceptions\ForbiddenException;
use Rowlinson\Api\Exceptions\NotFoundException;
use Rowlinson\Api\Exceptions\PricingServiceUnavailableException;
use Rowlinson\Api\Exceptions\UnauthorisedException;
use Rowlinson\Api\Exceptions\UnavailableException;
use Rowlinson\Api\Responses\ApiResponse;
use Symfony\Component\HttpFoundation\Response;

abstract class AbstractClient
{
    protected Credentials $credentials;
    protected Client $http;
    protected SerializerInterface $serializer;

    public function __construct(
        Credentials $credentials,
        Client $http,
        SerializerInterface $serializer
    ) {
        $this->credentials = $credentials;
        $this->http = $http;
        $this->serializer = $serializer;
    }

    /**
     * Convenience method for getting the response from a URL
     */
    protected function get(string $url): ApiResponse
    {
        return $this->request('GET', $url);
    }

    /**
     * Convenience method for posting a request ($body)
     *
     * @param mixed $body
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ParameterTypeHint.MissingNativeTypeHint
     */
    protected function post(string $url, $body = null): ApiResponse
    {
        return $this->request('POST', $url, $body);
    }

    /**
     * Convenience method for putting a request ($body)
     *
     * @param mixed $body
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ParameterTypeHint.MissingNativeTypeHint
     */
    protected function put(string $url, $body = null): ApiResponse
    {
        return $this->request('PUT', $url, $body);
    }

    /**
     * Convenience method for deleting a request ($body)
     *
     * @param mixed $body
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ParameterTypeHint.MissingNativeTypeHint
     */
    protected function delete(string $url, $body = null): ApiResponse
    {
        return $this->request('DELETE', $url, $body);
    }

    /**
     * @param mixed $body
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ParameterTypeHint.MissingNativeTypeHint
     */
    protected function request(string $method, string $url, $body = null): ApiResponse
    {
        $params = [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->credentials->getApiKey(),
                'Accept' => 'application/json',
            ],
        ];

        if (in_array(strtoupper($method), [ 'POST', 'PUT' ])) {
            $params['body'] = $this->serializer->serialize($body ?? [], 'json');
            $params['headers']['Content-Type'] = 'application/json';
        }

        try {
            // this uses Sdk::transformApiResponseMiddleware to create an ApiResponse
            return $this->http->request($method, $url, $params); // @phpstan-ignore-line
        } catch (ClientException $e) {
            if ($e->hasResponse()) {
                $json = json_decode((string)$e->getResponse()->getBody());

                switch ($e->getResponse()->getStatusCode()) {
                    case Response::HTTP_NOT_FOUND:
                        $description = is_object($json)
                            && property_exists($json, 'description') ? $json->description : '';

                        throw new NotFoundException($description, Response::HTTP_NOT_FOUND, $e);
                    case Response::HTTP_UNAUTHORIZED:
                        $description = is_object($json)
                            && property_exists($json, 'description') ? $json->description : '';

                        throw new UnauthorisedException($description, Response::HTTP_UNAUTHORIZED, $e);
                    case Response::HTTP_FORBIDDEN:
                        $description = is_object($json)
                            && property_exists($json, 'description') ? $json->description : '';

                        throw new ForbiddenException($description, Response::HTTP_FORBIDDEN, $e);
                    case Response::HTTP_CONFLICT:
                        $description = is_object($json)
                            && property_exists($json, 'description') ? $json->description : '';

                        throw new ConflictException($description, Response::HTTP_CONFLICT, $e);
                }
            }

            throw new UnavailableException('', ($e->hasResponse()) ? $e->getResponse()->getStatusCode() : 400, $e);
        } catch (ServerException $e) {
            if (
                $e->hasResponse() &&
                $e->getResponse()->getStatusCode() === Response::HTTP_SERVICE_UNAVAILABLE &&
                count($e->getResponse()->getHeader('X-Pricing-Unavailable')) > 0
            ) {
                throw new PricingServiceUnavailableException(
                    'Pricing service unavailable',
                    Response::HTTP_SERVICE_UNAVAILABLE,
                    $e
                );
            }

            throw $e;
        } catch (GuzzleException $e) {
            throw new UnavailableException('', 500, $e);
        }
    }
}
