<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use Rowlinson\Api\Collections\PaginatedCollection;
use Rowlinson\Api\Responses\Product\ProductCategoryResponse;
use Rowlinson\Api\Responses\Product\ProductResponse;

class ProductClient extends AbstractClient
{
    /**
     * Get the details of an individual product
     */
    public function getProduct(string $productAlias, ?bool $approvedVariants = null): ProductResponse
    {
        $queryParams = array_filter([
            'approved' => $approvedVariants,
        ]);
        $queryString = (count($queryParams) > 0) ? '?' . http_build_query($queryParams) : '';

        $response = $this->get('products/' . $productAlias . $queryString);
        /** @var ProductResponse $productResponse */
        $productResponse = $this->serializer->deserialize((string)$response->getBody(), ProductResponse::class, 'json');

        return $productResponse;
    }

    /**
     * Gets all products that are relevant for the logged in user
     *
     * @param array<string> $skus Filter products by ones with the supplied variants
     * @param array<string> $types Filter products by types
     * @return PaginatedCollection<ProductResponse>
     */
    public function getProducts(
        int $page = 1,
        ?string $categoryId = null,
        array $skus = [],
        array $types = [],
        bool $includeVariants = false,
        ?string $organisationAlias = null,
        ?string $schoolAlias = null
    ): PaginatedCollection {
        $response = $this->get('products?' . http_build_query([
            'page' => $page,
            'category' => $categoryId,
            'sku' => $skus,
            'type' => $types,
            'include_variants' => $includeVariants,
            'organisation' => $organisationAlias,
            'school' => $schoolAlias,
        ]));

        $productResponse = $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . ProductResponse::class . '>',
            'json'
        );

        return new PaginatedCollection(
            $productResponse,
            $page,
            $response->getPaginationPerPage(),
            $response->getPaginationTotalItems()
        );
    }

    /**
     * Get the number of products available for the provided query options
     *
     * @param array<string> $skus
     * @param array<string> $types
     */
    public function getProductCount(
        ?string $categoryId = null,
        array $skus = [],
        array $types = [],
        ?string $organisationAlias = null,
        ?string $schoolAlias = null
    ): int {
        $response = $this->get('products_count?' . http_build_query([
            'category' => $categoryId,
            'sku' => $skus,
            'type' => $types,
            'organisation' => $organisationAlias,
            'school' => $schoolAlias,
        ]));

        return intval((string)$response->getBody());
    }

    /**
     * Gets all product categories that are relevant for the logged in user
     *
     * @return PaginatedCollection<ProductCategoryResponse>
     */
    public function getProductCategories(int $page = 1, int $perPage = 10): PaginatedCollection
    {
        $response = $this->get('product_categories?' . http_build_query([
            'page' => $page,
            'per_page' => $perPage,
        ]));

        $productCategoryResponses = $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . ProductCategoryResponse::class . '>',
            'json'
        );

        return new PaginatedCollection(
            $productCategoryResponses,
            $page,
            $perPage,
            $response->getPaginationTotalItems()
        );
    }

    /**
     * Search for products - SKU and product name
     *
     * @return PaginatedCollection<ProductResponse>
     */
    public function search(string $query): PaginatedCollection
    {
        $response = $this->get('products?' . http_build_query([
            'search_term' => $query,
        ]));


        $productResponses = $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . ProductResponse::class . '>',
            'json'
        );

        return new PaginatedCollection($productResponses, 1, 10, $response->getPaginationTotalItems());
    }

    /**
     * Download images for a product, can be filtered to contain images for only certain colours
     *
     * @param array<string> $colours
     */
    public function downloadImages(string $productAlias, array $colours): void
    {
        $this->post('products/' . $productAlias . '/download_images', [
            'colours' => $colours,
        ]);
    }
}
