<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use Rowlinson\Api\Collections\PaginatedCollection;
use Rowlinson\Api\Exceptions\NotFoundException;
use Rowlinson\Api\Requests\User\FeedbackRequest;
use Rowlinson\Api\Requests\User\NewPasswordRequest;
use Rowlinson\Api\Requests\User\RegisterRequest;
use Rowlinson\Api\Requests\User\ResetPasswordRequest;
use Rowlinson\Api\Requests\User\UpdateRequest;
use Rowlinson\Api\Responses\User\ListAllSingleResponse;
use Rowlinson\Api\Responses\User\MeResponse;

class UserClient extends AbstractClient
{
    /**
     * Request a password reset for the user identified by $email
     */
    public function resetPassword(string $email, string $url): void
    {
        $this->post('auth/reset_password', new ResetPasswordRequest(
            $this->credentials->getClientKey(),
            $this->credentials->getClientSecret(),
            $email,
            $url
        ));
    }

    /**
     * From a password reset request, set a new password for a user
     */
    public function newPassword(string $code, string $password): bool
    {
        try {
            $this->post('auth/new_password', new NewPasswordRequest(
                $this->credentials->getClientKey(),
                $this->credentials->getClientSecret(),
                $code,
                $password
            ));
        } catch (NotFoundException $e) {
            return false;
        }

        return true;
    }

    /**
     * Get the currently authenticated user
     */
    public function getCurrentUser(): MeResponse
    {
        $resp = $this->get('users/me');

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize((string)$resp->getBody(), MeResponse::class, 'json');
    }

    /**
     * Update the currently authenticated user
     */
    public function updateCurrentUser(
        ?string $firstName,
        ?string $lastName,
        ?string $jobTitle,
        ?string $email,
        ?string $mobileNumber,
        ?string $landlineNumber,
        ?bool $enabled2FA,
        ?string $password
    ): void {
        $this->put('users/me', new UpdateRequest(
            $firstName,
            $lastName,
            $jobTitle,
            $email,
            $mobileNumber,
            $landlineNumber,
            $enabled2FA,
            $password
        ));
    }

    /**
     * Register a new user via invite code
     */
    public function register(string $code, string $password, ?string $phoneMobile): bool
    {
        try {
            $this->post('users/invite/' . $code, new RegisterRequest(
                $password,
                $phoneMobile
            ));
        } catch (NotFoundException $e) {
            return false;
        }

        return true;
    }

    /**
     * List all users in the system with optional filtering by name and email address
     *
     * @return PaginatedCollection<ListAllSingleResponse>
     */
    public function listAll(?string $name, ?string $email, int $page): PaginatedCollection
    {
        $response = $this->get('users?' . http_build_query(array_filter([
            'page' => $page > 0 ? $page : 1,
            'email' => ($email !== null) ? trim($email) : null,
            'name' => ($name !== null) ? trim($name) : null,
        ])));

        $users = $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . ListAllSingleResponse::class . '>',
            'json'
        );

        return new PaginatedCollection(
            $users,
            $page,
            $response->getPaginationPerPage(),
            $response->getPaginationTotalItems()
        );
    }

    /**
     * Remove a user
     */
    public function remove(string $email): void
    {
        $this->delete('users/remove/' . $email);
    }

    /**
     * Submit user feedback
     */
    public function feedback(int $rating, string $uniqueId, ?string $feedback): void
    {
        $this->post('users/feedback', new FeedbackRequest($rating, $uniqueId, $feedback));
    }
}
