<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use Rowlinson\Api\Collections\PaginatedCollection;
use Rowlinson\Api\Exceptions\NotFoundException;
use Rowlinson\Api\Requests\Invite\InviteRequest;
use Rowlinson\Api\Responses\Invites\GetInviteResponse;
use Rowlinson\Api\Responses\Invites\InviteResponse;

class InviteClient extends AbstractClient
{
    /**
     * List all pending invites
     *
     * @return PaginatedCollection<InviteResponse>
     */
    public function listAll(): PaginatedCollection
    {
        $response = $this->get('users/invites');

        /** @var array<InviteResponse> $invites */
        $invites = $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . InviteResponse::class . '>',
            'json'
        );

        return new PaginatedCollection($invites, 1, count($invites), count($invites));
    }

    /**
     * Invite a new user to the system
     */
    public function invite(
        string $firstName,
        string $lastName,
        ?string $jobTitle,
        string $email,
        string $url
    ): void {
        $this->post('users/invite', new InviteRequest(
            $firstName,
            $lastName,
            $jobTitle,
            $email,
            $url
        ));
    }

    /**
     * Get an invite by its code
     */
    public function getInvite(string $code): ?GetInviteResponse
    {
        $resp = null;

        try {
            $resp = $this->get('users/invite/' . $code);
        } catch (NotFoundException $e) {
            return null;
        }

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize((string)$resp->getBody(), GetInviteResponse::class, 'json');
    }

    /**
     * Resend an invite to an $email
     */
    public function resendInvite(string $email): void
    {
        $this->post('users/invite', new InviteRequest(
            '  ', // first name
            '  ', // last name
            null, //job title
            $email,
            '  ', // url
        ));
    }
}
