<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use Rowlinson\Api\Exceptions\UnauthorisedException;
use Rowlinson\Api\Requests\Auth\ConfirmNew2faRequest;
use Rowlinson\Api\Requests\Auth\ImpersonateRequest;
use Rowlinson\Api\Requests\Auth\LoginRequest;
use Rowlinson\Api\Requests\Auth\SendNew2faRequest;
use Rowlinson\Api\Requests\Auth\SsoRequest;
use Rowlinson\Api\Requests\Auth\VerifyCredentialsRequest;
use Rowlinson\Api\Requests\Auth\VerifyRequest;
use Rowlinson\Api\Responses\Auth\ConfirmNew2faResponse;
use Rowlinson\Api\Responses\Auth\LoginResponse;
use Rowlinson\Api\Responses\Auth\SendNew2FaResponse;

class AuthClient extends AbstractClient
{
    /**
     * Set the API key that the credentials use
     *
     * Only makes sense in AuthClient as it deals with authentication and impersonation
     */
    public function setApiKey(string $apiKey): void
    {
        $this->credentials->setApiKey($apiKey);
    }

    /**
     * Log in with the supplied credentials
     */
    public function logIn(string $email, string $password, ?string $role = null): ?LoginResponse
    {
        $resp = null;

        try {
            $resp = $this->post('auth/login', new LoginRequest(
                $this->credentials->getClientKey(),
                $this->credentials->getClientSecret(),
                $email,
                $password,
                $role
            ));
        } catch (UnauthorisedException $e) {
            return null;
        }

        /** @var LoginResponse */
        return $this->serializer->deserialize((string)$resp->getBody(), LoginResponse::class, 'json');
    }

    /**
     * Exchange an SSO token for an API key
     */
    public function sso(string $token): ?LoginResponse
    {
        $resp = null;

        try {
            $resp = $this->post('auth/sso', new SsoRequest(
                $this->credentials->getClientKey(),
                $this->credentials->getClientSecret(),
                $token
            ));
        } catch (UnauthorisedException $e) {
            return null;
        }

        /** @var LoginResponse */
        return $this->serializer->deserialize((string)$resp->getBody(), LoginResponse::class, 'json');
    }

    /**
     * Destroy the API key of the currently logged in user
     */
    public function logout(): void
    {
        $this->post('auth/logout');
    }

    /**
     * Verify a 2FA challenge
     */
    public function verify(string $challengeId, string $token): ?LoginResponse
    {
        $resp = null;

        try {
            $resp = $this->post('auth/verify', new VerifyRequest(
                $this->credentials->getClientKey(),
                $this->credentials->getClientSecret(),
                $challengeId,
                $token
            ));
        } catch (UnauthorisedException $e) {
            return null;
        }

        /** @var LoginResponse */
        return $this->serializer->deserialize((string)$resp->getBody(), LoginResponse::class, 'json');
    }

    /**
     * Verify the credentials of the currently logged in user
     */
    public function verifyCredentials(string $email, string $password, ?string $role): bool
    {
        try {
            $this->post('auth/verify_credentials', new VerifyCredentialsRequest(
                $email,
                $password,
                $role
            ));

            return true;
        } catch (UnauthorisedException $e) {
        }

        return false;
    }

    /**
     * Sends a 2fa code to $mobile number
     */
    public function send2fa(string $mobile): string
    {
        $response = $this->post('auth/send_2fa', new SendNew2faRequest($mobile));

        /** @var SendNew2FaResponse $response */
        $response = $this->serializer->deserialize(
            (string)$response->getBody(),
            SendNew2FaResponse::class,
            'json'
        );

        return $response->challenge_id;
    }

    /**
     * Verify that we have the correct $code for a given $challenge
     */
    public function confirm2fa(string $challenge, string $code): bool
    {
        $response = $this->post('auth/confirm_2fa', new ConfirmNew2faRequest($challenge, $code));

        /** @var ConfirmNew2faResponse $response */
        $response = $this->serializer->deserialize(
            (string)$response->getBody(),
            ConfirmNew2faResponse::class,
            'json'
        );

        return $response->success;
    }

    /**
     * Impersonate the user with the supplied email address
     */
    public function impersonate(string $email): LoginResponse
    {
        $body = new ImpersonateRequest($email);
        $response = $this->post('auth/impersonate', $body);

        /** @var LoginResponse */
        return $this->serializer->deserialize((string)$response->getBody(), LoginResponse::class, 'json');
    }

    /**
     * Get the current (if any) customer API key
     */
    public function getCustomerApiKey(): LoginResponse
    {
        $response = $this->get('auth/api_keys');

        /** @var LoginResponse */
        return $this->serializer->deserialize((string)$response->getBody(), LoginResponse::class, 'json');
    }

    /**
     * Generate customer API key, if a key already exists it will be regenerated
     */
    public function generateCustomerApiKey(): LoginResponse
    {
        $response = $this->post('auth/api_keys');

        /** @var LoginResponse */
        return $this->serializer->deserialize((string)$response->getBody(), LoginResponse::class, 'json');
    }
}
