<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use Concat\Http\Middleware\Logger;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\CurlHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Response;
use JMS\Serializer\GraphNavigatorInterface;
use JMS\Serializer\Handler\HandlerRegistry;
use JMS\Serializer\SerializerInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Rowlinson\Api\Credentials\Credentials;
use Rowlinson\Api\Responses\ApiResponse;
use Rowlinson\Api\Responses\School\SchoolResponse;
use RuntimeException;

class Sdk
{
    public const BASE_URI = 'https://api.rowlinsons.test/api/v1/';

    private Credentials $credentials;
    private SerializerInterface $serializer;
    private Client $http;
    /** @var array<AbstractClient> */
    private array $clients = [];

    /**
     * @param Credentials $credentials The credentials used to access services
     * @param SerializerInterface $serializer For (de)serializing requests and responses
     * @param string $baseUri The base URI of the API, defaults to the BASE_URI constant
     * @param LoggerInterface|null $logger PSR-3 logger
     */
    public function __construct(
        Credentials $credentials,
        SerializerInterface $serializer,
        string $baseUri = self::BASE_URI,
        ?LoggerInterface $logger = null,
        int $timeout = 10
    ) {
        $this->credentials = $credentials;
        $this->serializer = $serializer;

        $handler = new CurlHandler();
        $stack = HandlerStack::create($handler);
        $stack->push(Middleware::mapResponse($this->transformApiResponseMiddleware()));

        if ($logger !== null) {
            $stack->push(new Logger($logger)); // @phpstan-ignore-line
        }

        $this->http = new Client([
            'base_uri' => $baseUri,
            'timeout' => $timeout,
            'connect_timeout' => $timeout,
            'read_timeout' => $timeout,
            'handler' => $stack,
        ]);
    }

    public static function serializerHandlers(): \Closure
    {
        return function (HandlerRegistry $registry): void {
            $registry->registerHandler(
                GraphNavigatorInterface::DIRECTION_DESERIALIZATION,
                SchoolResponse::class,
                'json',
                function ($visitor, $stringOrArray): SchoolResponse {
                    $resp = new SchoolResponse();

                    if (is_string($stringOrArray)) {
                        $resp->name = $stringOrArray;
                    } elseif (is_array($stringOrArray)) {
                        $resp->name = $stringOrArray['name'] ?? null;
                        $resp->alias = $stringOrArray['alias'] ?? null;
                    }

                    return $resp;
                }
            );
        };
    }

    /**
     * Get the user client
     */
    public function createUser(): UserClient
    {
        $client = $this->getClient('user');

        if (!($client instanceof UserClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    /**
     * Get the order client
     */
    public function createOrder(): OrderClient
    {
        $client = $this->getClient('order');

        if (!($client instanceof OrderClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    /**
     * Get the products client
     */
    public function createProduct(): ProductClient
    {
        $client = $this->getClient('product');

        if (!($client instanceof ProductClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    /**
     * Get the basket client
     */
    public function createBasket(): BasketClient
    {
        $client = $this->getClient('basket');

        if (!($client instanceof BasketClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    /**
     * Get the delivery locations client
     */
    public function createDeliveryLocation(): DeliveryLocationClient
    {
        $client = $this->getClient('deliveryLocation');

        if (!($client instanceof DeliveryLocationClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    public function createAuth(): AuthClient
    {
        $client = $this->getClient('auth');

        if (!($client instanceof AuthClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    /**
     * Get the invite client
     */
    public function createInvite(): InviteClient
    {
        $client = $this->getClient('invite');

        if (!($client instanceof InviteClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    /**
     * Get the organisation client
     */
    public function createOrganisation(): OrganisationClient
    {
        $client = $this->getClient('organisation');

        if (!($client instanceof OrganisationClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    /**
     * Get the leavers hoodies client
     */
    public function createLeaversHoodie(): LeaversHoodiesClient
    {
        $client = $this->getClient('leaversHoodies');

        if (!($client instanceof LeaversHoodiesClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    public function createSchool(): SchoolClient
    {
        $client = $this->getClient('school');

        if (!($client instanceof SchoolClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    public function createBankHolidays(): BankHolidaysClient
    {
        $client = $this->getClient('bankHolidays');

        if (!($client instanceof BankHolidaysClient)) {
            throw new RuntimeException();
        }

        return $client;
    }

    /**
     * Create and locally cache an API client
     */
    protected function getClient(string $key): AbstractClient
    {
        if (!array_key_exists($key, $this->clients)) {
            $clientClass = __NAMESPACE__ . '\\' . ucfirst($key) . 'Client';

            if (!class_exists($clientClass) || !is_subclass_of($clientClass, AbstractClient::class)) {
                throw new \RuntimeException($key . ' / ' . $clientClass . ' does not extend AbstractClient');
            }

            $this->clients[$key] = new $clientClass(
                $this->credentials,
                $this->http,
                $this->serializer
            );
        }

        return $this->clients[$key];
    }

    /**
     * Transform a standard Guzzle Response object into our extended ApiResponse object
     */
    private function transformApiResponseMiddleware(): callable
    {
        return function (ResponseInterface $response): ResponseInterface {
            if ($response instanceof Response) {
                return ApiResponse::createFromGuzzleResponse($response);
            }

            return $response;
        };
    }
}
