<?php

declare(strict_types=1);

namespace Rowlinson\Api\Exceptions;

use GuzzleHttp\Exception\RequestException;
use Psr\Http\Message\RequestInterface;

/**
 * The API is unavailable (network issues, 500 response, timed out, unknown response format)
 */
class UnavailableException extends ApiException
{
    /**
     * Get the HTTP method that was used for the API request
     */
    public function getMethod(): ?string
    {
        if (!($this->getPrevious() instanceof RequestException)) {
            return null;
        }

        return (string)$this->getPrevious()->getRequest()->getMethod();
    }

    /**
     * Get the full URL that was requested
     */
    public function getUrl(): ?string
    {
        if (!($this->getPrevious() instanceof RequestException)) {
            return null;
        }

        return (string)$this->getPrevious()->getRequest()->getUri();
    }

    /**
     * Get all of the headers sent with the API request
     *
     * @return array<string>
     */
    public function getRequestHeaders(): array
    {
        if (!($this->getPrevious() instanceof RequestException)) {
            return [];
        }

        $headers = $this->getPrevious()->getRequest()->getHeaders();
        $ret = [];

        foreach ($headers as $name => $values) {
            foreach ($values as $value) {
                $ret[] = $name . ': ' . $value;
            }
        }

        return $ret;
    }

    /**
     * Get the body of the request sent to the API
     *
     * @return mixed
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ReturnTypeHint.MissingNativeTypeHint
     */
    public function getRequestBody()
    {
        if (!($this->getPrevious() instanceof RequestException)) {
            return null;
        }

        /** @var RequestInterface $request */
        $request = $this->getPrevious()->getRequest();
        $body = (string)$request->getBody();

        // convert to a JSON object if possible
        if (strpos($request->getHeaderLine('Content-Type'), 'application/json') !== false) {
            $jsonBody = json_decode($body);
            if (json_last_error() === JSON_ERROR_NONE) {
                $body = $jsonBody;
            }
        }

        return $body;
    }

    /**
     * Get the response status code if available
     */
    public function getStatusCode(): ?int
    {
        if (!($this->getPrevious() instanceof RequestException) || $this->getPrevious()->getResponse() === null) {
            return null;
        }

        return $this->getPrevious()->getResponse()->getStatusCode();
    }

    /**
     * Get any headers that the server responded to the request with
     *
     * @return array<string>
     */
    public function getResponseHeaders(): array
    {
        if (!($this->getPrevious() instanceof RequestException) || $this->getPrevious()->getResponse() === null) {
            return [];
        }

        $headers = $this->getPrevious()->getResponse()->getHeaders();
        $ret = [];

        foreach ($headers as $name => $values) {
            foreach ($values as $value) {
                $ret[] = $name . ': ' . $value;
            }
        }

        return $ret;
    }

    /**
     * Get the response body if available
     *
     * @return mixed
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ReturnTypeHint.MissingNativeTypeHint
     */
    public function getResponseBody()
    {
        if (!($this->getPrevious() instanceof RequestException) || $this->getPrevious()->getResponse() === null) {
            return null;
        }

        $response = $this->getPrevious()->getResponse();
        $body = (string)$response->getBody();

        // convert to a JSON object if possible
        if (strpos($response->getHeaderLine('Content-Type'), 'application/json') !== false) {
            $jsonBody = json_decode($body);
            if (json_last_error() === JSON_ERROR_NONE) {
                $body = $jsonBody;
            }
        }

        return $body;
    }
}
