<?php

declare(strict_types=1);

namespace Rowlinson\Api\Collections;

use ArrayAccess;
use ArrayIterator;
use Countable;
use IteratorAggregate;

/**
 * @template T
 * @implements ArrayAccess<int|string,T>
 * @implements IteratorAggregate<int|string,T>
 * @phpstan-consistent-constructor
 */
class PaginatedCollection implements ArrayAccess, IteratorAggregate, Countable
{
    /** @var array<int|string,T> */
    protected array $items;
    protected int $page;
    protected int $perPage;
    protected int $total;

    /**
     * @param array<int|string,T> $items
     * @param int $page The page this collection is of the total
     * @param int $perPage How many items per page was requested
     * @param int $total The total number of items if this collection weren't paginated
     */
    public function __construct(array $items = [], int $page = 1, int $perPage = 10, int $total = 1)
    {
        $this->items = $items;
        $this->page = $page;
        $this->perPage = $perPage;
        $this->total = $total;
    }

    public function getPage(): int
    {
        return $this->page;
    }

    public function getPerPage(): int
    {
        return $this->perPage;
    }

    public function getTotal(): int
    {
        return $this->total;
    }

    public function getPageCount(): int
    {
        return intval(ceil($this->total / $this->perPage));
    }

    /**
     * @template T2 The mapped collection's type
     * @param callable(T): T2 $callback
     * @return PaginatedCollection<T2>
     */
    public function map(callable $callback): self
    {
        $keys = array_keys($this->items);
        $items = array_map($callback, $this->items, $keys);

        return new static(array_combine($keys, $items), $this->page, $this->perPage, $this->total);
    }

    /**
     * @param int|string $offset
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ParameterTypeHint.MissingNativeTypeHint
     */
    public function offsetExists($offset): bool
    {
        return array_key_exists($offset, $this->items);
    }

    /**
     * @param int|string $offset
     * @return T
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ParameterTypeHint.MissingNativeTypeHint
     */
    #[\ReturnTypeWillChange]
    public function offsetGet($offset)
    {
        /** @var T */
        return $this->items[$offset];
    }

    /**
     * @param int|string $offset
     * @param T $value
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ParameterTypeHint.MissingNativeTypeHint
     */
    public function offsetSet($offset, $value): void
    {
        $this->items[$offset] = $value;
    }

    /**
     * @param int|string $offset
     * @phpcsSuppress SlevomatCodingStandard.TypeHints.ParameterTypeHint.MissingNativeTypeHint
     */
    public function offsetUnset($offset): void
    {
        unset($this->items[$offset]);
    }

    /**
     * @return ArrayIterator<int|string,T>
     */
    public function getIterator(): ArrayIterator
    {
        return new ArrayIterator($this->items);
    }

    public function count(): int
    {
        return count($this->items);
    }
}
