<?php

declare(strict_types=1);

namespace Rowlinson\Api;

use DateTimeImmutable;
use Rowlinson\Api\Collections\PaginatedCollection;
use Rowlinson\Api\Collections\SortedPaginatedCollection;
use Rowlinson\Api\Requests\Order\DeliveryLocation;
use Rowlinson\Api\Requests\Order\ExportOrdersRequest;
use Rowlinson\Api\Requests\Order\Item;
use Rowlinson\Api\Requests\Order\PlaceOrderRequest;
use Rowlinson\Api\Requests\Order\SplitRequest;
use Rowlinson\Api\Requests\Order\UpdateOrderTypeRequest;
use Rowlinson\Api\Responses\Order\OrderResponse;
use Rowlinson\Api\Responses\Order\OrdersResponse;
use Rowlinson\Api\Responses\Order\OrderTypeResponse;
use Rowlinson\Api\Responses\Order\SplitResponse;

class OrderClient extends AbstractClient
{
    /**
     * Place a new order with the supplied details, ignoring the current basket
     *
     * @param array<Item> $items
     */
    public function placeOrder(
        array $items,
        ?string $deliveryLocationId,
        ?DeliveryLocation $newDeliveryLocation,
        ?string $deliveryInstructions,
        ?string $customerReference,
        ?string $orderType,
        ?string $deduplicationKey,
        ?string $amountPaid = null
    ): void {
        $headers = [];
        if ($deduplicationKey !== null) {
            $headers['Deduplication-Key'] = $deduplicationKey;
        }

        $this->post(
            'orders/place',
            new PlaceOrderRequest(
                $items,
                $deliveryLocationId,
                $newDeliveryLocation,
                $deliveryInstructions,
                $customerReference,
                $orderType,
                $amountPaid
            ),
            $headers
        );
    }

    /**
     * Get the orders for the organisation of the currently logged in user
     *
     * @return SortedPaginatedCollection<OrdersResponse>
     */
    public function getOrders(
        int $page = 1,
        string $sortBy = 'updated',
        string $sortDir = 'DESC'
    ): SortedPaginatedCollection {
        $params = [
            'page' => $page,
            'sort_by' => $sortBy,
            'sort_dir' => $sortDir,
        ];

        $response = $this->get('orders?' . http_build_query($params));

        $orderResponses = $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . OrdersResponse::class . '>',
            'json'
        );

        return new SortedPaginatedCollection(
            $response->getSortedBy(),
            $response->getSortDirection(),
            $orderResponses,
            $page,
            $response->getPaginationPerPage(),
            $response->getPaginationTotalItems()
        );
    }

    /**
     * Request orders between the supplied $start and $finish (inclusive by date) are exported as a CSV
     *
     * @param array<string>|null $orderTypes
     */
    public function exportOrders(
        DateTimeImmutable $start,
        DateTimeImmutable $finish,
        string $type,
        ?array $orderTypes
    ): void {
        $this->post('orders/export', new ExportOrdersRequest($start, $finish, $type, $orderTypes));
    }

    /**
     * Get the details of an individual order
     */
    public function getOrder(string $orderNumber): ?OrderResponse
    {
        $response = $this->get('orders/' . $orderNumber);

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize((string)$response->getBody(), OrderResponse::class, 'json');
    }

    /**
     * Get the available forward order types for the currently authenticated user
     *
     * @param bool $all Whether to retrieve all order types, only available for admin users
     * @param bool $historical Whether to retrieve the order types a customer has in their order history
     * @return PaginatedCollection<OrderTypeResponse>
     */
    public function getOrderTypes(bool $all = false, bool $historical = false): PaginatedCollection
    {
        $params = [];

        if ($all) {
            $params['all'] = 1;
        }
        if ($historical) {
            $params['historical'] = 1;
        }

        $qs = ($all || $historical) ? '?' . http_build_query($params) : '';
        $response = $this->get('orders/types' . $qs);

        /** @var array<OrderTypeResponse> $orderTypeResponses */
        $orderTypeResponses = $this->serializer->deserialize(
            (string)$response->getBody(),
            'array<' . OrderTypeResponse::class . '>',
            'json'
        );

        return new PaginatedCollection(
            $orderTypeResponses,
            1,
            count($orderTypeResponses),
            count($orderTypeResponses)
        );
    }

    public function updateOrderType(
        string $alias,
        ?DateTimeImmutable $dueDate,
        ?bool $visible,
        ?DateTimeImmutable $cutOffTime,
        ?int $workingDays
    ): void {
        $this->put('orders/types/' . $alias, new UpdateOrderTypeRequest($dueDate, $visible, $cutOffTime, $workingDays));
    }

    /**
     * @param array<string> $skuAliases
     */
    public function split(?string $orderType, array $skuAliases): SplitResponse
    {
        $response = $this->post('orders/split', new SplitRequest($orderType, $skuAliases));

        // @phpstan-ignore-next-line
        return $this->serializer->deserialize(
            (string)$response->getBody(),
            SplitResponse::class,
            'json'
        );
    }
}
