<?php

App::uses('Component', 'Controller');
App::import('Lib', 'EvTransactions.GatewayInterface');

include('../../Vendor/autoload.php');

use Omnipay\Omnipay as Omnipay;
use Omnipay\Common\CreditCard as OmnipayCreditCard;

/*
 Worldpay gateway settings are;

 array(
	'installationId' => '',			// the id that exists per merchant account for worldpay
	'accountId' => '',				// i dont THINK this is used, as the installationId above is used instead
	'secretWord' => '',
	'callbackPassword' => '',
	'testMode' => false				// this wants to be true if we're in testing mode
);

 */
class WorldpayOffsiteComponent extends Component implements GatewayInterface {

	/**
	 * the controller that this transaction component is associated with
	 * @var EvCoreController
	 */
	private $__controller = null;

	/**
	 * the gateway itself, in this case, an Omnipay Worldpay instance
	 * @var \Omnipay\WorldPay\Gateway
	 */
	protected $_api = null;

	/**
	 * array of parameters that is set and then sent to gateway purchase method
	 * @var array
	 */
	protected $_params = [];

	/**
	 * the credit card being passed to the payment method
	 * @var \Omnipay\Common\CreditCard
	 */
	protected $_card = null;

	/**
	 * Initialise the gateway with a controller and any other stuff you want to happen initially
	 */
	public function initialize(Controller $controller) {

		parent::initialize($controller);

		$this->__controller = $controller;

		$this->_api = Omnipay::create('WorldPay');
	}

	/**
	 * init
	 *
	 * Use the function setup and connection / settings that need setting up in the gateway
	 */
	public function setup() {
		if (Configure::read('db.config') == 'live') {
			$this->_config = Configure::read('EvWorldpayOffsite.live');
			$this->_api->setTestMode(false);
		} else {
			$this->_config = Configure::read('EvWorldpayOffsite.dev');
			$this->_api->setTestMode(true);
		}
	}

	/**
	 * setupPayment
	 *
	 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
	 *
	 * @param int - transaction id we have created
	 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
	 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
	 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
	 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
	 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
	 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = array()) {
		$this->_params = [];

		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transactionId;

		$this->_card = new OmnipayCreditCard();

		if (isset($extra['billing_address']['name'])) {
			$fullName = $extra['billing_address']['name'];
			$splitName = explode(' ', trim($fullName));
			$lastName = array_pop($splitName);

			$this->_card->setLastName($lastName);

			if (count($splitName)) {
				$this->_card->setFirstName(implode(' ', $splitName));
			} else {
				$this->_card->setFirstName('');
			}
		} else {
			$this->_card->setFirstName((isset($extra['user']['User']['first_name']) ? $extra['user']['User']['first_name'] : $extra['user']['User']['name']));

			if (isset($extra['user']['User']['last_name'])) {
				$this->_card->setLastName($extra['user']['User']['last_name']);
			}
		}

		if (isset($extra['user']['User']['email'])) {
			$this->_card->setEmail($extra['user']['User']['email']);
		}

		$this->_card->setCountry('GB');
		if (isset($extra['billing_address']['address1'])) {
			$this->_card->setAddress1($extra['billing_address']['address1']);
		}
		if (isset($extra['billing_address']['address2'])) {
			$this->_card->setAddress2($extra['billing_address']['address2']);
		}
		if (isset($extra['billing_address']['city'])) {
			$this->_card->setCity($extra['billing_address']['city']);
		}
		if (isset($extra['billing_address']['post_code'])) {
			$this->_card->setPostcode($extra['billing_address']['post_code']);
		}

		//Set currency
		if (is_array($amount)) {
			$currencyCode = $amount['currency'];
			$amount = $amount['amount'];
		} else {

			$currencyCode = Configure::read('Transactions.currency');
		}

		$this->_params = array(
			'transactionId' => $transactionId . "-" . time(),
			'transactionReference' => 'test',
			'returnUrl' => $return['return'] . '?transaction=' . $transactionId,
			'currency' => $currencyCode,
			'amount' => $amount,
			'description' => 'Payment to ' . Configure::read('SiteSetting.site_title'),
			'clientIp' => $this->__controller->request->clientIp()
		);

		$this->_params = array_merge($this->_params, $this->_config);

		// $this->__controller->Session->write('Transactions.SagepayDirect.params', $params);
		//$this->__controller->Session->write('Transactions.Worldpay.params', $this->_params);

		// this should be the new card variable, declared above the array
		$this->_params['card'] = $this->_card;

	}

	/**
	 * getPayment
	 *
	 * Everything should be setup, actually take the payment
	 *
	 * @param int transactions id
	 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
	 */
	public function getPayment($transactionId) {
		try {
			$response = $this->_api->purchase($this->_params)->send();
			if ($response->isSuccessful()) {
				// mark order as complete
				// probably something like taking the transaction id from the response, and saving it
				// to our own transaction object?
			} elseif ($response->isRedirect()) {
				$response->redirect();
			} else {
				// display error to customer
				exit($response->getMessage());
			}
		} catch (\Exception $e) {
			debug($e);
			// internal error, log exception and display a generic message to the customer
			die('Sorry, there was an error processing your payment. Please try again later.');
		}
	}

	/**
	 * return
	 *
	 * deal with a return from the gateway and check for success / fail
	 *
	 * @return array - with three elements,
	 *				 - 'result' = true/false value
	 *  			 - 'message' = text message about transaction (i.e. reason for failing)
	 * 				 - 'transaction_id' = int of the transaction row
	 */
	public function processReturn() {
		// Check if config needs to be loaded
		if ($this->_config == null) {
			$this->setup();
		}

		// CakeLog::write('error', 'Processing return');
		// CakeLog::write('error', Router::url( $this->here, true ));
		// CakeLog::write('error', print_r($this->__controller->request->data, true));
		// CakeLog::write('error', print_r($this->_config, true));

		$response = $this->__controller->request->data;

		// CakeLog::write('error', print_r($response, true));
		// CakeLog::write('error', print_r($this->_config['callbackPassword'], true));

		// Setup defaults
		$message = '';
		$result = false;

		//Extra the transaction ID
		$transactionId = (int)(substr($response['cartId'], 0, strpos($response['cartId'], '-')));

		// Check we have a sensible Transaction ID
		if (!$transactionId > 0) {
			throw new Exception('Invalid transaction ID' . print_r($transactionId, true));
		}

		// Confirm it is a genuine callback by verifying the callback payment set in Worldpay Admin
		if ($response['callbackPW'] == $this->_config['callbackPassword']) {
			if ($response['transStatus'] == 'Y') {
				$result = true;
			} else {
				$result = false;
				$message = 'There was a problem processing payment' . $response['rawAuthMessage'];
			}

		} else {
			CakeLog::write('error', 'Worldpay: Invalid callback password used' . $this->_config['callbackPassword'] . ' != ' . $response['callbackPW']);
			$result = false;
		}

		return array(
			'transaction_id' => $transactionId,
			'result' => $result,
			'message' => $message
		);
	}

}
