<?php

App::uses('Component', 'Controller');
use Omnipay\Omnipay as Omnipay;

class SagePayComponent extends Component
{
	private $__controller = null;

	protected $_api = null;

	protected $_data = null;

	protected $_error_url = null;

	/**
	 * Sets a reference to the calling controller from within the component.
	 *
	 * @see Component::initialize()
	 */
	public function initialize(Controller $controller) {

		parent::initialize($controller);

		$this->__controller = $controller;

		$this->_api = Omnipay::create('SagePay_Direct');
	}

	/**
	 * init
	 *
	 * Use the function setup and connection / settings that need setting up in the gateway
	 */
	public function setup() {

		if (Configure::read('db.config') == 'live') {

			$this->_config = Configure::read('EvSagePay.SagePayDirect.live');

		} else {

			$this->_config = Configure::read('EvSagePay.SagePayDirect.dev');

			$this->_api->setTestMode(true);

		}
		$this->_api->setVendor($this->_config['vendor']);
	}

	/**
	 * setupPayment
	 *
	 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
	 *
	 * @param int - transaction id we have created
	 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
	 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
	 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
	 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
	 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
	 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = array()) {
		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transactionId;

		$return['return'] = $return['return'] . '?transaction=' . $transactionId;

		if (strpos($extra['CreditCard']['card_holder_name'], ' ') !== false) {
			$name = explode(' ', $extra['CreditCard']['card_holder_name'], 2);
			$firstName = $name[0];
			$lastName = $name[1];
		} else {
			$firstName = $extra['CreditCard']['card_holder_name'];
			$lastName = ' . ';
		}

		$cardData = array(
			'firstName' => $firstName,
			'lastName' => $lastName,
			'number' => $extra['CreditCard']['card_number'],
			'startMonth' => (!empty($extra['CreditCard']['start_month'])) ? $extra['CreditCard']['start_month'] : null,
			'startYear' => (!empty($extra['CreditCard']['start_year'])) ? $extra['CreditCard']['start_year'] : null,
			'expiryMonth' => $extra['CreditCard']['expiry_month'],
			'expiryYear' => $extra['CreditCard']['expiry_year'],
			'cvv' => $extra['CreditCard']['security_number'],
			'address1' => $extra['delivery']['Address']['address1'],
			'address2' => $extra['delivery']['Address']['address2'],
			'city' => $extra['delivery']['Address']['city'],
			'postcode' => $extra['delivery']['Address']['post_code'],
			'state' => '',
			'country' => 'GB',
			'email' => $extra['user']['User']['email'],
			'phone' => ''
		);

		//Set currency
		if (is_array($amount)) {
			$currencyCode = $amount['currency'];
		} else {
			$currencyCode = Configure::read('EvSagePay.currency');
		}

		$params = array(
			'transactionId' => $transactionId . "-" . time(),
			'transactionReference' => 'test',
			'returnUrl' => $return['return'],
			'currency' => $currencyCode,
			'amount' => $amount['amount'],
			'description' => 'Payment to ' . Configure::read('SiteSetting.site_title'),
			'clientIp' => $this->__controller->request->clientIp()
		);

		if ($this->__controller->Session->read('EvSagePay.SagepayDirect.accountType') == 'M') {
			$params['accountType'] = 'M';
		}

		$this->__controller->Session->write('EvSagePay.params', $params);

		$params['card'] = $cardData;

		// Store the customers details on the transaction as we'll need them when they return.


		$this->_data = $this->_api->purchase(
			$params
		);
	}

	/**
	 * getPayment
	 *
	 * Everything should be setup, actually take the payment
	 *
	 * @param int transactions id
	 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
	 */
	public function getPayment($transactionId) {
		$result = $this->_sendRequest($this->_data);

		return array(
			'result' => $result,
			'transaction_id' => $transactionId
		);
	}

	/**
	 * return
	 *
	 * deal with a return from the gateway and check for success / fail
	 *
	 * @return array - with three elements,
	 *				 - 'result' = true/false value
	 *  			 - 'message' = text message about transaction (i.e. reason for failing)
	 * 				 - 'transaction_id' = int of the transaction row
	 */
	public function processReturn() {
		return $this->process3dSecure();
	}

	public function process3dSecure() {
		$params = $this->__controller->Session->read('EvSagePay.params');

		$request = $this->_api->completePurchase($params);

		$result = $this->_sendRequest($request);

		return array(
			'result' => $result,
			'transaction_id' => $params['transactionId']
		);
	}

	protected function _sendRequest($request) {
		$params = $this->__controller->Session->read('EvSagePay.params');

		$this->Transaction = ClassRegistry::init('EvTransactions.Transaction');
		$this->Transaction->id = substr($params['transactionId'], 0, strpos($params['transactionId'], '-'));

		try {

			$response = $request->send();

			//Customer is successfully paid
			if ($response->isSuccessful()) {
				// load the transactions model and update with the token
				$this->Transaction->save(array(
					'Transaction' => array(
						'payment_token' => $response->getTransactionReference(),
						'status' => 'success'
					)
				));
				$result = true;

			} elseif ($response->isRedirect()) {

				$response->redirect(); // this will automatically forward the customer. For 3D secure etc.

			} else {

				$this->Transaction->save(array(
					'Transaction' => array(
						'status' => 'failed',
						'message' => $response->getMessage()
					)
				));
				$result = false;
			}

		} catch(Exception $e) {

			// load the transactions model and update with the token
			$this->Transaction->save(array(
				'Transaction' => array(
					'status' => 'failed',
					'message' => $e->getMessage()
				)
			));
			$result = false;
		}

		return $result;
	}
}
