<?php
App::uses('AppComponent', 'Controller/Component');

class WishlistComponent extends AppComponent {

/**
 * Add a wishlist item to a user's wishlist. Gets the user id from the Auth component so only effects the
 * currently logged in user.
 *
 * @param string $model   The name of the model of the wishlist item.
 * @param int    $modelId The id of the wishlist item.
 */
	public function addItem($model, $modelId) {
		$userId = $this->_controller->Auth->user('User.id');

		if (empty($userId) || empty($model) || empty($modelId)) {
			return;
		}

		$this->_controller->loadModel('EvWishlist.WishlistItem');

		//Check if the user already has this item in their wishlist
		$wishlistItem = $this->_controller->WishlistItem->getItemForUser($model, $modelId, $userId);

		if (!empty($wishlistItem)) {
			return $wishlistItem;
		}

		return $this->_controller->WishlistItem->save([
			'user_id' => $userId,
			'model' => $model,
			'model_id' => $modelId,
		]);
	}

/**
 * Remove a wishlist item from a user's wishlist. Gets the user id from the Auth component so only effects the
 * currently logged in user.
 *
 * @param string $model   The name of the model of the wishlist item.
 * @param int    $modelId The id of the wishlist item.
 */
	public function deleteItem($model, $modelId) {
		$userId = $this->_controller->Auth->user('User.id');

		if (empty($userId) || empty($model) || empty($modelId)) {
			return;
		}

		$this->_controller->loadModel('EvWishlist.WishlistItem');

		//Check if the user already has this item in their wishlist
		$wishlistItem = $this->_controller->WishlistItem->getItemForUser($model, $modelId, $userId);

		if (empty($wishlistItem)) {
			//The currently logged in user doesn't have this item in their wishlist.
			return false;
		}

		return $this->_controller->WishlistItem->delete($wishlistItem['WishlistItem']['id']);
	}

/**
 * Get a redirect for when adding/removing an item from the wishlist.
 *
 * @param string $redirectName The name of the redirect you want to get. "add" or "delete". Check config for others.
 * @return array|string A url array or a url string.
 */
	public function getRedirect($redirectName) {
		$referer = $this->_controller->referer();

		if (empty($this->_controller->Auth->user())) {
			if (empty($this->_controller->Session->read('EvWishlist.beforeLogin'))) {
				$this->_controller->Session->write('EvWishlist.beforeLogin', $referer);
			}
			$this->_controller->Session->write('Auth.redirect', $this->_controller->request->here());
			$this->_controller->redirect($this->_controller->Auth->loginAction);
		}

		if (!empty($this->_controller->Session->read('EvWishlist.beforeLogin'))) {
			$beforeLogin = $this->_controller->Session->read('EvWishlist.beforeLogin');
			$this->_controller->Session->delete('EvWishlist.beforeLogin');

			if (Configure::check('EvWishlist.redirects.afterLogin')) {
				return Configure::read('EvWishlist.redirects.afterLogin');
			}

			return $beforeLogin;
		}

		if (Configure::check('EvWishlist.redirects.' . $redirectName)) {
			return Configure::read('EvWishlist.redirects.' . $redirectName);
		}

		return $referer;
	}

/**
 * Get a wishlist item for the currently logged in user using the model and model id.
 *
 * @param string $model   The name of the model of the item to get for the user.
 * @param int    $modelId The model id of the item to get for the user.
 * @return void|array A wishlist item if one is found otherwise void.
 */
	public function getItemForUser($model, $modelId) {
		$userId = $this->_controller->Auth->user('User.id');

		if (empty($userId) || empty($model) || empty($modelId)) {
			return;
		}

		$this->_controller->loadModel('EvWishlist.WishlistItem');
		return $this->_controller->WishlistItem->getItemForUser($model, $modelId, $userId);
	}
}
