<?php

	App::uses('VebraAppModel', 'Vebra.Model');
	App::uses('Branch', 'Vebra.Model');
	App::uses('Property', 'Vebra.Model');

	/**
	 * Description of Vebra
	 *
	 * @author Luke
	 */
	class Vebra extends VebraAppModel {

		const FILE_TYPE_IMAGE = 0;
		const FILE_TYPE_FLOORPLAN = 2;
		const FILE_TYPE_BROCHURE = 7;
		const FILE_TYPE_EPC = 9;

		public $useDbConfig = 'vebra';
		public $useTable = false;
		public $hasOne = array('Vebra.VebraToken');
		public $hasMany = array('Vebra.VebraLog');
		private $uploadedImages = array();

		/**
		 * Look for an active token in the database and return it or return false.
		 * Note, this will only return the token - not the whole array.
		 * We don't cache the value because it's time sensitive, and storing the expiry time in three places isn't a good idea
		 * @return boolean|string
		 */
		public function getActiveToken() {
			$active_token = $this->VebraToken->findActive();
			return ($active_token !== false && !empty($active_token) ? $active_token['VebraToken']['token'] : false);
		}

		/**
		 * Get all branches from the API. VebraApi returns and array of branches, each with the following fields:
		 * - name
		 * - branchid
		 * - firmid
		 * - url
		 * - street
		 * - town
		 * - county
		 * - postcode
		 * - phone
		 * - email
		 */
		public function getBranches() {
			$this->_initialiseConnection(); // get a connection
			$branches = $this->getDataSource()->sendRequest('branch', $this->getActiveToken()); // send the request (returned in array format)
			if (is_array($branches['branch']) && !empty($branches['branch'])) { // if the response is a non-empty array
				$branches_with_details = array();
				foreach ($branches['branch'] as $branch) { // for each branch found
					preg_match('/\/([0-9]+)$/', $branch['url'], $matches); // get the branch ID from the end of the URL. Yeah, you'd think you could just use the branch id we get back, wouldn't you? 					$details = $this->_getBranchDetails($branch['url']); // get the branch details
					$details['BranchID'] = $branch['id'] = $matches[1]; // set both branch ID's (yeah, different cases, for laughs) to the one at the end fo the URL
					$branches_with_details[] = array_change_key_case(array_merge($branch, $details), CASE_LOWER); // merge the branch with the branch details (keys to lower case so we don't get dupes)
				}
				$this->_logRequest('get_all_branches');
				return $branches_with_details;
			}
			return false;
		}

		/**
		 * Takes a full URL (because the API returns the full URL for 'convenience') and gets
		 * a full set of details for a branch. Typically used when looping through branches
		 * @param string $url
		 * @return type
		 */
		protected function _getBranchDetails($url) {
			return $this->getDataSource()->sendRequest($url, $this->getActiveToken());
		}

		public function getProperties($branchId) {
			$this->_initialiseConnection();
			$properties = $this->getDataSource()->sendRequest('branch/' . $branchId . '/property', $this->getActiveToken());
			$this->_logRequest('get_branch_properties', $branchId);
			if (is_array($properties['property']) && !empty($properties['property'])) {
				foreach ($properties['property'] as &$property) { // set the branch id for each property
					preg_match('/branch\/([0-9]+)/', $property['url'], $matches);
					$property['branch_id'] = $matches[1];
				}
				return $this->_importProperties($properties);
			}
			return false;
		}

		public function getUpdatedProperties() {
			$this->_initialiseConnection();

			$options = array(
				'conditions' => array(
					'action' => 'get_updated_properties' // we do this, because each request for branch properties is logged individually, and thus have different timestamps
				),
				'order' => 'created DESC'
			);
			$lastLog = $this->VebraLog->find('first', $options);
			if ($lastLog) {
				list($lastRanDate, $lastRanTime) = explode(' ', $lastLog['VebraLog']['created']);
				list($year, $month, $day) = explode('-', $lastRanDate);
				list($hours, $minutes, $seconds) = explode(':', $lastRanTime);
				$fromDate = implode('/', array($year, $month, $day, $hours, $minutes, $seconds));
			} else { // if we've never run the API before
				$fromDate = '1970/01/01/00/00/01'; // url segments separated by slashes in the format yyyy/mm/dd/hh/mm/ss
			}

			$properties = $this->getDataSource()->sendRequest('/property/' . $fromDate, $this->getActiveToken());

			if (!empty($properties['property'])) {
				$keys = array_keys($properties['property']);
				$topKey = array_shift($keys);
				if (!is_int($topKey)) { // top first key of the 'property' array isn't integer, meaning we're not dealing with an array of properties, but the one single proprety to update
					$data['property'][] = $properties['property'];
					$properties = $data;
				}
			}

			// The only difference in array data between this method and getting branch properties is the 'action'
			if (!empty($properties)) {
				$this->log('Property update overview (what will be updated / deleted / added):');
				$this->log($properties);
				foreach ($properties['property'] as $key => &$property) {
					switch ($property['action']) {
						case 'updated':
							preg_match('/branch\/([0-9]+)/', $property['url'], $matches);
							$branchId = $matches[1];
							$property['branch_id'] = $branchId;
							break;

						case 'deleted':
							$propertyModel = ClassRegistry::init('Vebra.Property');
							$dbProperty = $propertyModel->findById($property['propid']);
							// If we found a record in the database and it's not showing in the sold gallery already, then get rid of it
							if ($dbProperty && !$propertyModel->showInSoldGallery($dbProperty)) {
								$propertyModel->delete($property['propid']);
							} else if ($dbProperty && $propertyModel->showInSoldGallery($dbProperty)) {
								$propertyModel->id = $property['propid'];
								$propertyModel->saveField('show_in_main_listings', false);
							}
							unset($properties['property'][$key]); // so that we don't try to import it again
							break;
					}
				}
			}
			if (isset($properties['property']) && is_array($properties['property']) && !empty($properties['property'])) {
				$return = $this->_importProperties($properties);
			} else {
				$return = null;
			}

			$this->_logRequest('get_updated_properties');
			return $return;
		}

		public function getProperty($branchId, $propertyId) {
			$this->_initialiseConnection();
			return $this->getDataSource()->sendRequest('/branch/' . $branchId . '/property/' . $propertyId, $this->getActiveToken());
		}

		protected function _getPropertyDetails($url) {
			return $this->getDataSource()->sendRequest($url, $this->getActiveToken());
		}

		protected function _importProperties(array $properties) {
			$properties_with_details = array();
			$c = 0;

			foreach ($properties['property'] as $property) {
				if (!isset($property['url']) || empty($property['url'])) {
					$this->log('property with no url (property not imported): ');
					$this->log($property);
					continue;
				}
				$details = $this->_getPropertyDetails($property['url']);

				// Get the address details
				if (isset($details['address'])) {
					foreach ($details['address'] as $key => $value) {
						$property[$key] = $value;
					}
				}

				// Fix uploaded date, because they do it weird
				list($day, $month, $year) = explode('/', $details['uploaded']);
				$details['uploaded'] = implode('-', array($year, $month, $day));

				// Make sure no null values go into the database
				$details['bedrooms'] = (!isset($details['bedrooms']) || $details['bedrooms'] === null ? 0 : $details['bedrooms']);
				$details['receptions'] = (!isset($details['receptions']) || $details['receptions'] === null ? 0 : $details['receptions']);
				$details['bathrooms'] = (!isset($details['bathrooms']) || $details['bathrooms'] === null ? 0 : $details['bathrooms']);
				$details['garden'] = (!isset($details['garden']) || !$details['garden'] ? 0 : $details['garden']);
				$details['parking'] = (!isset($details['parking']) || !$details['parking'] ? 0 : $details['parking']);

				// Set the type
				$details['type'] = (isset($details['type'][0]) && !empty($details['type'][0]) ? $details['type'][0] : null);
				if (isset(Property::$typeMap[$details['type']])) {
					$details['display_type'] = Property::$typeMap[$details['type']]; // e.g. 'House - detached' => 'Detached house'
				} else {
					$details['display_type'] = $details['type'];
				}

				if (isset($details['web_status']) && in_array($details['web_status'], array(Property::STATUS_SSTC, Property::STATUS_VEBRA_SSTC))) {
					$property['date_moved_to_sstc'] = date('Y-m-d');
				}
				
				// Do we have a tour?
				$details['tour_url'] = $details['userfield1'];

				$property['id'] = (isset($property['prop_id']) ? $property['prop_id'] : (isset($property['propid']) ? $property['propid'] : null)); // use the Vebra prop_id as our primary key (to save useless auto-increment data)
				$properties_with_details[$c]['Property'] = array_change_key_case(array_merge($property, $details), CASE_LOWER);

				// Save the paragraphs (as associations in cake)
				if (!empty($details['paragraphs']['paragraph'])) {

					if (empty($details['paragraphs']['paragraph'][0])) {
						$details['paragraphs']['paragraph'] = array($details['paragraphs']['paragraph']);
					}
					
					foreach ($details['paragraphs']['paragraph'] as $paragraph) {
						// This shouldn'y cause a conflict with any other array keys, given the data that we have at the moment, but if you ever lose data at the paragraph level, look into this
						// Merge paragraph attributes into the paragraph itself
						if (isset($paragraph['@attributes'])) {
							foreach ($paragraph['@attributes'] as $key => $value) {
								$paragraph[$key] = $value;
							}
							unset($paragraph['@attributes']);
						}

						// Merge the dimensions into the paragraph itself
						if (isset($paragraph['dimensions'])) {
							foreach ($paragraph['dimensions'] as $key => $value) {
								if (is_array($value) || empty($value)) { // if the value is an array, skip it
									continue;
								}
								$paragraph['dimensions_' . $key] = $value;
							}
							unset($paragraph['dimensions']);
						}
						if (isset($paragraph['id'])) {
							unset($paragraph['id']); // We don't want to mess with the primary key auto incrementing
						}
						$properties_with_details[$c]['Paragraph'][] = $paragraph; // we add directly to the merged array because we need to keep the casing for 'Paragraph' (model associations)
					}
				}

				if (!empty($details['files']['file'])) {
					foreach ($details['files']['file'] as $file) {
						switch ($file['@attributes']['type']) {
							case static::FILE_TYPE_FLOORPLAN:
								if (!isset($file['name']) || is_array($file['name']) || empty($file['name'])) { // if there's no file name, set it to empty so the switch falls back to it's dfault case without throwing errors
									$file['name'] = '';
								}

								$imageType = 'Floorplan';
								if(strpos(strtolower($file['name']), 'town')!==false) {
									$imageType = 'Townmap';
								}
								elseif(strpos(strtolower($file['name']), 'street')!==false) {
									$imageType = 'Streetmap';
								}								
								elseif(strpos(strtolower($file['name']), 'detail')!==false) {
									$imageType = 'Detailmap';
								}									

								$this->_importImage($file, $properties_with_details[$c], $imageType);

								break;

							case static::FILE_TYPE_BROCHURE:
								// This is a PDF document, so clearly can't be uploaded in the same way as images
								$this->_importDocument($file, $properties_with_details[$c], 'Brochure');
								break;

							case static::FILE_TYPE_EPC:
								// EPCs are also just images, and there is typically only one of them, though don't restrict it as such
								$this->_importImage($file, $properties_with_details[$c], 'Epc');
								break;

							case static::FILE_TYPE_IMAGE:
							default:
								$this->_importImage($file, $properties_with_details[$c]);
								break;
						}
					}
				}

				$c++;
			}

			$this->log('Processed property data (for updates and insertions only):');
			$this->log($properties_with_details);

			return $properties_with_details;
		}

		protected function _importImage($image, &$property, $attachment_type = 'Image') {
			preg_match('/([^\/]+\.([a-zA-Z]{3,4}))$/', $image['url'], $matches);
			if (!isset($matches[1]) && isset($matches[2])) {
				$this->log('Unable to retrieve file data for an image returned by Vebra:');
				$this->log($image);
			}
			$filename = $matches[1];
			$extension = $matches[2];
			switch ($extension) {
				case 'jpg' || 'jpeg':
					$type = 'image/jpeg';
					break;

				case 'png':
					$type = 'image/png';
					break;

				case 'gif':
					$type = 'image/gif';
					break;

				default:
					$this->log('Attempting to import an image where the image type could not be determined. Logging image data:');
					$this->log($image);
					return false;
					break;
			}

			//check we haven't previosuly dealt with this image
			if(!in_array($filename, $this->uploadedImages)) { 

				// Store the temp image to upload shortly
				$tmp_name = Configure::read('Vebra.tmp_dir') . $filename;
				if (!is_dir(Configure::read('Vebra.tmp_dir'))) {
					mkdir(Configure::read('Vebra.tmp_dir'));
				}

				if (copy($image['url'], $tmp_name)) { // Add the image only if the import works
					// Store the image using the regular upload method (no need to do anything special with it)
					$this->uploadedImages[] = $filename;

					$imageData = array(
						'model' => 'Property',
						'filename' => array(
							'name' => $filename,
							'type' => $type,
							'size' => filesize($tmp_name),
							'tmp_name' => $tmp_name,
							'error' => 0,
						),
						'attachment_type' => ($attachment_type !== 'Image' ? $attachment_type . 'Image' : $attachment_type),
						'type' => $type,
					);
					$property['Image'][] = $imageData; // add image to the images array, ready for saving
				}
			}
			else {
				$this->log('Duplicate image detected');
			}
			return $property;
		}

		protected function _importDocument() {
			return true;
		}

		protected function _storeToken($token) {
			$this->VebraToken->create();
			$this->VebraToken->set('token', base64_encode($token));
			$this->VebraToken->save();
			$this->VebraToken->clear();
			return $this;
		}

		protected function _logRequest($action, $branchId = null, $propertyId = null, $paragraphId = null, $documentId = null) {
			$guzzleRequest = $this->getDataSource()->getGuzzleRequest();
			$url = $guzzleRequest->getUrl();
			$guzzleResponse = $this->getDataSource()->getGuzzleResponse();
			$guzzleException = $this->getDataSource()->getGuzzleException();

			$vebraLog = array();
			$vebraLog['VebraLog']['action'] = $action;
			$vebraLog['VebraLog']['branch_id'] = $branchId;
			$vebraLog['VebraLog']['property_id'] = $propertyId;
			$vebraLog['VebraLog']['paragraph_id'] = $paragraphId;
			$vebraLog['VebraLog']['document_id'] = $documentId;
			$vebraLog['VebraLog']['url'] = $url;
			$vebraLog['VebraLog']['request_header'] = json_encode($guzzleRequest->getHeaders());
			$vebraLog['VebraLog']['response_header'] = json_encode(($guzzleResponse !== null ? $guzzleResponse->getHeaders() : $guzzleException->getRequest()->getResponse()->getHeaders()));
			$vebraLog['VebraLog']['response_body'] = json_encode(($guzzleResponse !== null ? $guzzleResponse->getBody() : $guzzleException->getRequest()->getResponse()->getBody()));
			$vebraLog['VebraLog']['response_code'] = json_encode(($guzzleResponse !== null ? $guzzleResponse->getStatusCode() : $guzzleException->getRequest()->getResponse()->getStatusCode()));
			$this->VebraLog->create();
			$this->VebraLog->save($vebraLog);

			return $this;
		}

		/**
		 * Get a new active token if none exist currently.
		 * This can be pasted atop every method that needs authentication, and it'll take care of getting you a new token when needed.
		 * It works because we don't store the token in the class - we get it from the database each time. This ensures that the API doesn't fall over part-way through
		 * @return Vebra
		 */
		protected function _initialiseConnection() {
			if ($this->getActiveToken() === false) {
				$token = $this->getDataSource()->getNewToken();
				$this->_storeToken($token);
			}
			return $this;
		}

	}
