<?php

	use GuzzleHttp\Client;
	use GuzzleHttp\Exception\RequestException;

	/**
	 * Description of VebraApi
	 *
	 * @author Luke
	 */
	class VebraApi extends DataSource {

		protected $vebraConfig = null;
		protected $baseApiUrl = null;
		protected $reconnectionAttempts = 0;
		protected $timeBetweenReconnectionAttempts = 2;
		protected $guzzleRequest; // log the last request that we send here, so that whatever uses this data source can grab it and log it (events don't work in a datasource - we tried)
		protected $guzzleResponse; // same as request
		protected $guzzleException; // same as request

		/** @var GuzzleHttp\Client */
		protected $Guzzle = null;

		public function __construct($config = array()) {

			parent::__construct($config);

			if (!isset($config['username']) || null === $config['username']) {
				throw new Exception('You must specifiy a username when attempting to connecto to the Vebra API');
			}

			if (!isset($config['password']) || null === $config['password']) {
				throw new Exception('You must specifiy a password when attempting to connecto to the Vebra API');
			}

			if (!isset($config['feed_id']) || null === $config['feed_id']) {
				throw new Exception('You must specifiy a feed ID when attempting to connecto to the Vebra API');
			}

			$this->vebraConfig = Configure::read('Vebra');
			$this->Guzzle = new Client(array('base_url' => $this->buildApiUrl()));

			return $this;
		}

		public function getNewToken() {
			try {
				// Create request and add auth
				$request = $this->Guzzle->createRequest('GET', $this->buildApiUrl('branch'));
				$request->addHeader('Authorization', 'Basic ' . base64_encode($this->config['username'] . ':' . $this->config['password']));
				// Send request
				$response = $this->Guzzle->send($request);

				if ($response->getStatusCode() == 200) { // everything worked fine
					if (!$response->hasHeader('Token')) { // no token in the header (how!? did we accidentally call this method instead of sendRequest()?)
						$this->log('No token given in the header when attempting to grab a new token (logging request, then response)');
						throw new BadRequestException();
					} else {
						return $response->getHeader('Token'); // return the token
					}
				} else { // something went wrong, because Guzzle throws exceptions in this case ordinarily
					$this->log('Guzzle failed to throw an exception when the status code of a Vebra request was ' . $response->getStatusCode());
					throw new BadRequestException();
				}
			} catch (RequestException $e) { //something other than a 200 should trigger an exception. There are more specific exceptions that can be caught if needs be - see Guzzle docs
				$this->_handleGuzzleException($e);

				if ($this->reconnectionAttempts < $this->vebraConfig['max_reconnection_attempts']) { // Retry for a new token (there could be a few second disparity in expiry times)
					if ($this->vebraConfig['time_between_reconnection_attempts'] !== null && is_integer($this->vebraConfig['time_between_reconnection_attempts'])) { // Sleep for (by default) two seconds before attempting to reconnect
						sleep($this->vebraConfig['time_between_reconnection_attempts']);
					}
					$this->reconnectionAttempts++;
					return $this->getNewToken();
				} else {
					$this->reconnectionAttempts = 0;
					throw new BadRequestException();
				}
			}
			return false; // shouldn't get here as everything above should return something
		}

		public function sendRequest($action, $token) {
			App::uses('CakeEvent', 'Event');
			App::uses('VebraApiListener', 'Vebra.Event');
			// Set guzzle variables to null, pending the next request
			$this->_setGuzzleException(null);
			$this->_setGuzzleRequest(null);
			$this->_setGuzzleResponse(null);

			try {
				if (is_array($action)) {
					throw new Exception('Array passed as request URL for Vebra request. $action must be either a URL or relative path from the base URL');
				}
				$url = (preg_match('/(http:\/\/|\.)/', $action) ? $action : $this->buildApiUrl($action)); // if we passed in a URL, just use that, otherwise build the route
				$request = $this->Guzzle->createRequest('GET', $url);
				$this->_setGuzzleRequest($request);
				$request->addHeader('Authorization', 'Basic ' . $token);

				$response = $this->Guzzle->send($request);
				$this->_setGuzzleResponse($response);

				return $this->_convertXMLResponseToArray($response->xml());
			} catch (RequestException $e) { //something other than a 200 should trigger an exception. There are more specific exceptions that can be caught if needs be - see Guzzle docs
				$this->_handleGuzzleException($e);
				throw new BadRequestException();
			} catch (Exception $e) { // rethrow exception when we get something other than a RequestException
				$this->log('Error thrown of type ' . get_class($e) . ' as part of a Guzzle request. Generally Guzzle should throw a RequestException.');
				throw $e;
			}
			return false; // shouldn't get here as something above should return something
		}

		public function buildApiUrl($action = null) {
			$base_url = $this->vebraConfig['api_url'];
			$api_url = str_replace(array('$feed_id', '$api_version'), array($this->config['feed_id'], $this->vebraConfig['api_version']), $base_url);
			if (null !== $action) {
				$api_url = rtrim($api_url, '/') . '/' . ltrim($action, '/');
			}
			return $api_url;
		}

		public function getGuzzleRequest() {
			return $this->guzzleRequest;
		}

		public function getGuzzleResponse() {
			return $this->guzzleResponse;
		}

		public function getGuzzleException() {
			return $this->guzzleException;
		}

		protected function _setGuzzleRequest($guzzleRequest) {
			$this->guzzleRequest = $guzzleRequest;
			return $this;
		}

		protected function _setGuzzleResponse($guzzleResponse) {
			$this->guzzleResponse = $guzzleResponse;
			return $this;
		}

		protected function _setGuzzleException(RequestException $guzzleException = null) {
			$this->guzzleException = $guzzleException;
			return $this;
		}

		/**
		 * Handle a Guzzle BadRequest exception. Generally speaking, we just need to log things here, as the getNewToken and sendRequest methods both want to do different things.
		 * That is, getting a new token will retry several times to get a new token, in case the one on the server hasn't yet been expired, whereas a typical request wont get a retry
		 * @param \GuzzleHttp\Exception\RequestException $e
		 * @return boolean
		 */
		protected function _handleGuzzleException(RequestException $e) {
			$this->_setGuzzleException($e);
			if ($e->hasResponse()) {
				switch ($e->getResponse()->getStatusCode()) { // Remember: Switch statements do loose-matching, so whilst the code returned is actually a string we can check against an int
					case 401: // generally this will be the issue. We're either giving the wrong user/pass, or are requesting a token when we already have one
						$this->log('401 Unauthorized error thrown when attempting to get new token from Vebra. This can occur when bad user/pass data is given, or when a token is already active');

						break;

					case 404:
						$this->log('Attempting to call a URL that does not exist in the Vebra API (' . $e->getRequest()->getUrl() . ')');
						break;

					default: // any other code falls to here
						$this->log('HTTP status code ' . $e->getResponse()->getStatusCode() . ' thrown by Vebra API');
						$this->log($e->getResponse());
						$this->log($e->getMessage());
						break;
				}
			}
			return true;
		}

		/**
		 * Convert a Guzzle-converted XML repsonse (thus, a SimpleXMLElement) to an array (easier for storing in CakePHP)
		 * Yes, I stole this code from the internet because I'm lazy; read up on it here (though it's nothing spectacular):
		 * http://stackoverflow.com/a/7778950/1370031
		 * @param SimpleXMLElement $xml
		 * @return array
		 */
		protected function _convertXMLResponseToArray(SimpleXMLElement $xml) {
			return json_decode(json_encode((array) $xml), TRUE);
		}

	}
