<?php
/**
 * Translatable Behaviour Class
 *
 * Adds ability to translate any stored database values on the fly.
 *
 * @author Rick Mills <rick@evoluted.net>
 */
class TranslatableBehavior extends ModelBehavior {

	/**
	 * Default values for settings.
	 *
	 * - models: Defines the models the item can be related to
	 *
	 * @access private
	 * @var array
	 */
	private $defaults = array(
		'models' => array(),
	);

	/**
	 * Globally excluded fields that we know we'll never, ever want to translate. These can't be
	 * overriden by models, but they can be added to. The primary purpose of this is to prevent
	 * users being presented with a bunch of fields they'll never want, or need to translate.
	 *
	 * Whilst it is possible to translate these fields, they are intentionally disabled from doing so.
	 * If you need this functionality, override the behaviour in your project, do NOT remove them
	 * from the plugin!
	 *
	 * @var array
	 */
	private $global_excluded_fields = array(
		'id',
		'is_active',
		'is_protected',
		'is_removed',
		'sequence',
		'created',
		'modified',
		'alias',
		'actual'
	);

	/**
	 * Globally exclude certain field types. It's extremely unlikely that anyone would ever want or
	 * need to translate an integer or date, and translating booleans would never be needed. This
	 * greatlysimplifies the amount of excluded fields that need to be entered in the individual
	 * model excluded fields array.
	 *
	 * Feel free to add obvious records to this list should any new ones be added, or if I've missed
	 * any off that should be here.
	 *
	 * @var array
	 */
	private $ignored_field_types = array(
		'boolean',
		'datetime',
		'integer',
		'select',
		'multiselect',
		'map',
		'map_data',
		'google_address', // No longer supported in EvForm however kept here for backwards compatability.
		'latitude',  // No longer supported in EvForm however kept here for backwards compatability.
		'longitude', // No longer supported in EvForm however kept here for backwards compatability.
		'hidden',
		'password',
		'float',
		'decimal',
		'slug',
		'route-alias'
	);

	private $global_excluded_models = array(
		'Route',
		'MetaData',
	);

	/**
	 * Set up the model behaviour
	 *
	 * This is where we set up all the settings needed for this behavior, such as which fields to
	 * exclude from our rules.
	 *
	 * @param  Model  $Model    Model object
	 * @param  array  $settings Individual model settings
	 */
	public function setup(Model $Model, $settings = array()) {

		// Set the default settings up which we'll add to in a moment.
		$defaults = array(
			'excluded_fields' => $this->global_excluded_fields,
			'excluded_models' => $this->global_excluded_models,
			'excluded_types' => $this->ignored_field_types
		);

		// Check to see if the model has any additional excluded fields set.
		if (isset($settings['excluded_fields']) && ! empty($settings['excluded_fields'])) {
			// Looks like it does, let's add those to the excluded fields array.
			$defaults['excluded_fields'] = array_merge($settings['excluded_fields'], $defaults['excluded_fields']);
		}

		// Check to see if the model has any additional excluded models set.
		if (isset($settings['excluded_models']) && ! empty($settings['excluded_models'])) {
			// Looks like it does, let's add those to the excluded models array.
			$defaults['excluded_models'] = array_merge($settings['excluded_models'], $defaults['excluded_models']);
		}

		// Set the settings globally so we can access them outside of the setup method.
		$this->settings = $defaults;

		// Add the global list of translated phrases to the settings - this gets created in the
		// AppController so that we're only loading it once to save on queries.
		$this->settings['phrases'] = Configure::read('translation.phrases');
	}

	/**
	 * AfterFind - used to manipulate result sets with our tranlated strings.
	 *
	 * @param  Model   $model  Model Object
	 * @param  array  $results  Result data set
	 * @param  boolean  $primary  Whether this model is being queried directly (vs. being queried as an association)
	 * @return  mixed  An array value will replace the value of $results - any other value will be ignored.
	 */
	public function afterFind(Model $model, $results, $primary = false) {
		$modelName = $model->alias;

		if (in_array($modelName, $this->settings['excluded_models'])) {
			return $results;
		}

		// Check if we're in the front end, and running a translated language. If this is the case
		// then we'll switch out this models values with the translated versions.
		$active_lang = CakeSession::read('EvTranslation.language_id');

		$this->TranslationPhrase = ClassRegistry::init('EvTranslation.TranslationPhrase');

		// Create an empty array where we'll store all the translation lookup queries. This saves us
		// needing to do multiple queries and reduces load time significantly.
		$query_params = array();

		foreach ($results as $result_id => $result_set) {

			if (is_array($result_set)) {

				foreach ($result_set as $model_name => $model_data) {

					if (! isset($model_data['id'])) {
						continue;
					}

					// Store the query params.
					$query_params[$model_name][] = $model_data['id'];

				}
			}
		}

		$cacheName = 'translation_phrases_';

		if (! empty($query_params)) {

			// Build the conditions
			$conditions = array();

			foreach ($query_params as $model => $ids) {

				// Add the model params to the conditions array.
				$conditions['OR'][]['AND'] = array(
					'model' => $model,
					'model_id' => $ids
				);

				if (is_array($ids)){
					$model_ids = implode('-', $ids);
				} else {
					$model_ids = $ids;
				}

				$cacheName . $model. '_'.$model_ids;

			}

			// Use the new conditions array to run a single query to pull all the data we need.
			$phrases = Cache::read($cacheName, 'long');
			if (!$phrases) {
				$phrases = $this->TranslationPhrase->find('all', array(
					'conditions' => $conditions,
					'contain' => array(
						'TranslationTranslation'
					)

				));
			}
			Cache::write($cacheName, $phrases, 'long');

			if (! empty($phrases)) {

				// Loop the phrases (should contain the phrase and the translation records)
				foreach ($phrases as $phrase) {

					if (isset($phrase['TranslationTranslation']) && ! empty($phrase['TranslationTranslation'])) {

						// Loop the translations for this phrase.
						foreach ($phrase['TranslationTranslation'] as $translation) {

							// Add the translated phrase data to the results array.
							if ($translation['translation'] !='') {
								$results[$result_id][$phrase['TranslationPhrase']['model']]['lang_' . $translation['translation_language_id'] . '_' . $phrase['TranslationPhrase']['model'] . '_' . $phrase['TranslationPhrase']['model_id'] . '_' . $phrase['TranslationPhrase']['model_field']] = $translation['translation'];
							}

							if ($active_lang == $translation['translation_language_id'] &&
								$modelName == $phrase['TranslationPhrase']['model'] &&
								$translation['translation'] !='') {

								$results[$result_id][$phrase['TranslationPhrase']['model']][$phrase['TranslationPhrase']['model_field']] = $translation['translation'];
							}

						}
					}
				}

			}

		}

		$results = $this->translateArray($results, $modelName);

		return $results;
	}


	/**
	 * After Save
	 *
	 * Adds or updates the translation phrases and translated data for this model.
	 *
	 * @param  Model  $model   Model that we're working with
	 * @param  boolean $created If true, this is a new record, if false we're updating an existing one.
	 * @param  array  $options Options passed from Model::save().
	 * @return bool
	 */
	public function afterSave(Model $model, $created, $options = array()) {
		if (!isset($model->data[$model->alias])) {
			return true;
		}

		$model_data = $model->data[$model->alias];

		$this->TranslationPhrase = ClassRegistry::init('TranslationPhrase');
		$this->TranslationTranslation = ClassRegistry::init('TranslationTranslation');

		$phrases = array();

		$cacheName = 'translation_phrases_' . $model->alias . '-' . $model->id;

		$phrase_records = Cache::read($cacheName, 'long');
		if (!$phrases) {

			$phrase_records = $this->TranslationPhrase->find('all', array(
				'conditions' => array(
					'model' => $model->alias,
					'model_id' => $model->id
				)
			));
		}
		Cache::write($cacheName, $phrase_records, 'long');

		if (! empty($phrase_records)) {

			foreach ($phrase_records as $phrase_record) {
				$phrases[$phrase_record['TranslationPhrase']['model'] . '_' . $phrase_record['TranslationPhrase']['model_id'] . '_' . $phrase_record['TranslationPhrase']['model_field']] = $phrase_record;
			}
		}

		// Loop the model data so we can get all fields
		foreach ($model_data as $field => $value) {

			// Check each field value to see if its a language string.
			if (strstr($field, 'lang_')) {

				// Reset the models so we don't conflict with any previous runs.
				$this->TranslationPhrase->clear();
				$this->TranslationTranslation->clear();

				// The field is a language translation. Now we need to extract the field name and
				// the language id from the variable. These are stored as lang_<LANGID>_<MODEL>_<MODELID>_<FIELD>
				$lang_var = ltrim($field, 'lang_');
				$lang_var = explode("_", $lang_var, 4);

				// Trim the model alias from the front.
				$lang_var[1] = ltrim($lang_var[1], $model->alias . '_');

				$language_id = $lang_var[0];
				$model_name = $lang_var[1];
				$model_id = $lang_var[2];
				$lang_field = $lang_var[3];

				if (array_key_exists($model->alias . '_' . $model_id . '_' . $lang_field, $phrases)) {

					// Phrase exists
					$phrase = $phrases[$model->alias . '_' . $model_id . '_' . $lang_field];
				} else {

					$phrase_data = array(
						'model' => $model->alias,
						'model_id' => $model->id,
						'model_field' => $lang_field,
						'translation_phrase_group_id' => null,
						'is_hidden' => '1',
						'is_active' => '1'
					);

					$phrase = $this->TranslationPhrase->save($phrase_data);
					$phrases[$model->alias . '_' . $model_id . '_' . $lang_field] = $phrase;
				}

				// Now check to see if we've already stored a translation. If we have, we'll
				// overwrite it. Otherwise, we'll create it.
				$translation = $this->TranslationTranslation->find('first', array(
					'conditions' => array(
						'translation_language_id' => $language_id,
						'translation_phrase_id' => $phrase['TranslationPhrase']['id']
					)
				));

				// Check to see if we've got a translation record
				if (!empty($translation)) {

					// A record exists! Delete it!
					$this->TranslationTranslation->delete($translation['TranslationTranslation']['id']);
				}

				// Now create a new record if the value is not empty.
				if ($value !='') {

					$translation_data = array(
						'translation_language_id' => $language_id,
						'translation_phrase_id' => $phrase['TranslationPhrase']['id'],
						'translation' => $value
					);

					// Save it
					$this->TranslationTranslation->save($translation_data);
				}
			}
		}
		// All done!
		return true;
	}

	/**
	 * After Delete
	 *
	 * Used to remove any orphaned translation records once a model item is deleted.
	 *
	 * @param  Model  $model Model that we're working with
	 * @return bool
	 */
	public function afterDelete(Model $model) {
		// Load the two translation models we're working with
		$this->TranslationPhrase = ClassRegistry::init('TranslationPhrase');
		$this->TranslationTranslation = ClassRegistry::init('TranslationTranslation');

		// Load up the applicable phrases
		$phrases = $this->TranslationPhrase->find('all', array(
			'conditions' => array(
				'TranslationPhrase.model' => $model->alias,
				'TranslationPhrase.model_id' => $model->id
			)
		));

		// Loop the phrases so we can get the individual related translation(s) for that phrase.
		foreach ($phrases as $phrase) {

			// Load all translations for this phrase
			$translations = $this->TranslationTranslation->find('all', array(
				'conditions' => array(
					'TranslationTranslation.translation_phrase_id' => $phrase['TranslationPhrase']['id']
				)
			));

			// Loop through the translation results
			foreach ($translations as $translation) {

				// Delete the translated phrase
				$this->TranslationTranslation->delete($translation['TranslationTranslation']['id']);
			}

			// Delete the phrase record as it's no longer needed/wanted.
			$this->TranslationPhrase->delete($phrase['TranslationPhrase']['id']);
		}

		// All done!
		return true;
	}

	/**
	 * Translate Array
	 *
	 * Used to replace the values of an array with it's translated values
	 *
	 * @param  Array $results  Results that we'll be translating
	 * @param  Object $model   Model object we'll be translating on
	 * @return Array  Returns the modified result set with the translated values
	 */
	public function translateArray($results, $model) {
		$new_result = array();

		if (is_object($model)) {
			$model_name = $model->alias;
		} else {
			$model_name = $model;
		}

		foreach ($results as $result => $fields) {

			if (is_array($fields)) {
				$sub_model = $model;

				if (!is_int($result)) {
					$sub_model = $result;
				}

				$new_result[$result] = $this->translateArray($fields, $sub_model);

			} elseif (isset($results['id'])) {

				$field_slug = $model . '_' . $results['id'] . '_' . $result;

				if (isset($this->settings['phrases']) && array_key_exists($field_slug, $this->settings['phrases'])) {

					// A translation for this field exists. Replace the result value with the
					// translated one.
					if ($this->settings['phrases'][$field_slug] !='') {

						$new_result[$result] = $this->settings['phrases'][$field_slug];
					} else {
						$new_result[$result] = $results[$result];
					}
				} else {
					$new_result[$result] = $results[$result];
				}

			}
		}

		return $new_result;
	}
}
