<?php
App::uses('Folder', 'Utility');
App::uses('File', 'Utility');
/**
 * TranslationUtil
 *
 * This class is used to store the active translations, and provides numerous key functions of the
 * translation plugin
 *
 * @author Rick Mills <rick@evoluted.net>
 */
class TranslationUtil
{
	public static $lang_id = 1; // Default language id
	public static $lang = array(); // Setup empty array for the active language data
	public static $default_lang = array(); // Setup empty array for the default language data

	public static $phrases = array(); // Setup empty array for the language phrases

	public static $TranslationLanguage; // Setup var for where we'll store the translation language model
	public static $TranslationPhrase; // Setup var for where we'll store the translation phrase model

	public static $request; // Stores the request data from the active controller method

	/**
	 * Init
	 *
	 * This is called by the controllers to initiate the translation system.
	 *
	 * @param  Object $request The current request data from the active controller method
	 */
	public static function init($request) {
		self::$request = $request; // Populate the gobal request var with the controller request data

		// Loads in the translation models we'll be using
		self::$TranslationLanguage = ClassRegistry::init('EvTranslation.TranslationLanguage');
		self::$TranslationPhrase = ClassRegistry::init('EvTranslation.TranslationPhrase');

		// Set the active language id
		self::$lang_id = Configure::read('EvTranslation.language_id');

		// Load the active language phrases

		//self::$lang = self::$TranslationLanguage->loadLanguage(self::$lang_id);
		self::$lang = self::loadCache(self::$lang_id);

		// Load the default language as a fallback option
		//self::$default_lang = self::$TranslationLanguage->loadDefaultLanguage();
		self::$default_lang = self::loadCache('default');

		// Write the language phrases into a config variable
		Configure::write('translation.phrases', self::$lang);

		// Check if we're running in the admin area. This is needed for the translatable behavior
		// to work out if it needs to run or not
		if (isset(self::$request->params['admin']) && self::$request->params['admin'] == '1') {
			$is_admin = true;
		} else {
			$is_admin = false;
		}

		// Write the admin status to a configure var just so it's a bit quicker to access elsewhere
		Configure::write('is_admin', $is_admin);
	}

	public static function loadCache($langId) {
		$file = new File(TMP . DS . 'translation_cache' . DS . 'cache_' . $langId . '.json');

		if ($file->exists()) {
			$fileData = $file->read();
			return json_decode($fileData, true);
		}
		return array();
	}

	/**
	 * Translate Fields
	 *
	 * Translates all fields in a data array. This can be uses to set up translations for a model
	 * on the fly.
	 *
	 * @param  array  $data  Model array of all fields that need to be translated
	 * @return array  Returns the modified data array containing the translated data fields
	 */
	public static function translateFields($data) {
		// Loop through the main data array
		foreach ($data as $key => $value) {

			// If the value contains an array it means it's got the model fields inside it.
			if (is_array($value)) {

				// Loop through the model fields
				foreach ($value as $model_field => $model_value) {

					// If the value is not an array, it means we've found the model value.
					if (!is_array($model_value)) {

						// Set up the vars we'll need to build the updated field language data.
						$modelName = $key;
						$modelId = $data[$key]['id'];
						$modelField = $model_field;

						if (isset(self::$lang[$modelName.'_'.$data[$key]['id'].'_'.$model_field]) && self::$lang[$modelName.'_'.$data[$key]['id'].'_'.$model_field] !='') {
							$data[$key][$model_field] = self::$lang[$modelName.'_'.$data[$key]['id'].'_'.$model_field];
						}
					}
					// TODO: Add support for sub-array translations.
				}
			}
		}
		return $data;
	}

	/**
	 * Translated Field
	 *
	 * Gets the translated field based on provided variables.
	 *
	 * @param  String  $model_name  The model name
	 * @param  Integer  $id  The model record id
	 * @param  String  $field_name  Name of the model field
	 * @return Mixed  Returns either the language value, or false if no value is found
	 */
	public static function translateField($model_name, $id, $field_name)
	{
		if (isset(self::$lang[$model_name.'_'.$id.'_'.$field_name]) && self::$lang[$model_name.'_'.$id.'_'.$field_name] != '') {
			return self::$lang[$model_name.'_'.$id.'_'.$field_name];
		}

		return false;
	}

	/**
	 * Translate By Slug
	 *
	 * Pulls an individual translation record based on a provided slug name.
	 *
	 * @param  String  $slug  Unique slug of the translation field
	 * @return  String  Returns the translated slug if one was found
	 */
	public static function translateBySlug($slug)
	{
		return self::$lang[$slug];
	}

	/**
	 * Admin Form Fields Translated
	 *
	 * This method creates the translation fields for the admin area editor.
	 *
	 * @param  Object $request  The request object from the controller
	 * @param  array  $fields  The field array that'll be supplied by the controlers _adminFormFields method
	 * @return Returns the array of translation fields
	 */
	public static function _adminFormFieldsTranslated($request, $fields = array())
	{
		$result = array();

		//pr($request->data);die();

		$language_model = ClassRegistry::init('EvTranslation.TranslationLanguage');

		$language_data = $language_model->find('all', array(
			'conditions' => array(
				'is_active' => 1,
				'is_default' => 0
			)
		));

		foreach ($language_data as $language) {

			$result[$language['TranslationLanguage']['id']] = array(
				'label' => $language['TranslationLanguage']['language'] . ' (' . $language['TranslationLanguage']['locale'] . ')',
				'fields' => array()
			);

			foreach ($request->data as $model_name => $model_rows) {

				if (empty($model_rows)) {
					continue;
				}

				if (! isset($model_rows[0]['id'])) {
					$model_rows = array($model_rows);
				}

				$model = ClassRegistry::init($model_name);

				$model_schema = $model->_schema;

				if ($model->Behaviors->loaded('Translatable')) {

					if (! isset($settings)) {
						$settings = $model->Behaviors->Translatable->settings;
					}

					foreach ($model_rows as $row_id => $row) {

						if (!isset($row['id'])) {
							continue;
						}

						foreach ($row as $field_name => $field_value) {
							$row_name_string = $model_name . '.lang_' . $language['TranslationLanguage']['id'] . '_' . $model_name . '_' . $row['id'] . '_' . $field_name;

							if(in_array($field_name, $settings['excluded_fields']) || !isset($model_schema[$field_name]['type']) || in_array($model_schema[$field_name]['type'], $settings['excluded_types'])) {
								continue;
							}

							$row_settings = $model_schema[$field_name];

							if (isset($model_schema[$field_name]['label'])) {
								$row_settings['label'] = $model_schema[$field_name]['label'];
							} else {
								$row_settings['label'] = InflectorExt::humanize($field_name);
							}

							$result[$language['TranslationLanguage']['id']]['fields'][$row_name_string] = $row_settings;

						}
					}
				}
			}
		}

		return $result;
	}

	/**
	 * Lang
	 *
	 * Used to load up a specific language slug, and falls back to the default language value should
	 * a translated value not be found. This is primarily used inside views to translate custom slugs.
	 *
	 * @param  String $slug The uniqye slug we'll be requesting a translation record for.
	 * @return String Returns either the translation record, the default record, or as a last resort, the slug.
	 */
	public static function lang($slug)
	{
		if (isset(self::$lang[$slug])) {

			return self::$lang[$slug];

		} elseif (isset(self::$default_lang[$slug])) {
			// The slug wasn't found in the active translations, so try loading it from the default
			// language.
			return self::$default_lang[$slug];

		} else {

			// TODO: Add an additional fallback to try and show the original model value instead of
			// the language slug. The downside with this is that it'll create extra queries, something
			// I'm trying to avoid.

			// Return the orginal slug as no translation was found.
			return $slug;
		}
	}
}
