<?php
/**
 * Translatable Behaviour Class
 *
 * Adds ability to translate any stored database values on the fly.
 *
 * @author Rick Mills <rick@evoluted.net>
 */
class TranslatableBehavior extends ModelBehavior {

    /**
     * Default values for settings.
     *
     * - models: Defines the models the item can be related to
     *
     * @access private
     * @var array
     */
    private $defaults = array(
        'models' => array(),
    );

    /**
     * Globally excluded fields that we know we'll never, ever want to translate. These can't be
     * overriden by models, but they can be added to. The primary purpose of this is to prevent
     * users being presented with a bunch of fields they'll never want, or need to translate.
     *
     * Whilst it is possible to translate these fields, they are intentionally disabled from doing so.
     * If you need this functionality, override the behaviour in your project, do NOT remove them
     * from the plugin!
     *
     * @var array
     */
    private $global_excluded_fields = array(
        'id',
        'is_active',
        'is_protected',
        'is_removed',
        'sequence',
        'created',
        'modified'
    );

    /**
     * Globally exclude certain field types. It's extremely unlikely that anyone would ever want or
     * need to translate an integer or date, and translating booleans would never be needed. This
     * greatlysimplifies the amount of excluded fields that need to be entered in the individual
     * model excluded fields array.
     *
     * Feel free to add obvious records to this list should any new ones be added, or if I've missed
     * any off that should be here.
     *
     * @var array
     */
    private $ignored_field_types = array(
        'boolean',
        'datetime',
        'integer',
        'select',
        'multiselect',
        'map',
        'map_data',
        'google_address', // No longer supported in EvForm however kept here for backwards compatability.
        'latitude',  // No longer supported in EvForm however kept here for backwards compatability.
        'longitude', // No longer supported in EvForm however kept here for backwards compatability.
        'hidden',
        'password',
        'float',
        'decimal',
        'slug',
        'route-alias'
    );

    /**
     * Set up the model behaviour
     *
     * This is where we set up all the settings needed for this behavior, such as which fields to
     * exclude from our rules.
     *
     * @param  Model  $Model    Model object
     * @param  array  $settings Individual model settings
     */
    public function setup(Model $Model, $settings = array())
    {

        // Set the default settings up which we'll add to in a moment.
        $defaults = array(
            'excluded_fields' => $this->global_excluded_fields
        );

        // Check to see if the model has any additional excluded fields set.
        if (isset($settings['excluded_fields']) && ! empty($settings['excluded_fields'])) {
            // Looks like it does, let's add those to the excluded fields array.
            $defaults['excluded_fields'] = array_merge($settings['excluded_fields'], $defaults['excluded_fields']);
        }

        // Set the settings globally so we can access them outside of the setup method.
        $this->settings = $defaults;

        // Add the global list of translated phrases to the settings - this gets created in the
        // AppController so that we're only loading it once to save on queries.
        $this->settings['phrases'] = Configure::read('translation.phrases');
    }

    /**
     * AfterFind - used to manipulate result sets with our tranlated strings.
     *
     * @param  Model   $model  Model Object
     * @param  array  $results  Result data set
     * @param  boolean  $primary  Whether this model is being queried directly (vs. being queried as an association)
     * @return  mixed  An array value will replace the value of $results - any other value will be ignored.
     */
    public function afterFind(Model $model, $results, $primary = false)
    {

        // Check if we're running in the admin area - if we are, we don't want to be doing this.
        $is_admin = Configure::read('is_admin');

        $this->TranslationPhrase = ClassRegistry::init('EvTranslation.TranslationPhrase');
        $this->TranslationLanguage = ClassRegistry::init('EvTranslation.TranslationLanguage');

        // NOTE: It's (extremely) like that there's a better way of doing this, but as it stands I
        // couldn't find a better way that wouldn't involve changes to other plugins.
        // This basically takes the meta data from the original model, which contains the model
        // names used by this model. We're going to strip out the priority field and the parent
        // model, then check our results array to see if the other models exist.
        if (isset($model->Behaviors->Meta->settings)) {
            $models = $model->Behaviors->Meta->settings;

            unset($models['priority']);
            //unset($models[$model->alias]);
        } else {
            $models = array();
        }

        // Load up all translation phrases for the model alias. We'll also contain the translations
        // and the language for each of those translations
        $all_phrases = $this->TranslationPhrase->find('all', array(
            'contain' => array(
                'TranslationTranslation',
                'TranslationTranslation.TranslationLanguage'
            ),
            'conditions' => array(
                'TranslationPhrase.model' => $model->alias
            ),
        ));

        // Reorganise the phrase results into a more sensible data set.
        $all_phrases = Hash::combine($all_phrases, "{n}.TranslationPhrase.model_id", "{n}");


        // Loop through the results.
        foreach ($results as $id => $result) {
            // Check that the field type allows for translations (generally you'll never need/want
            // to translate boolean or integer/numeric fields as these arent really translatable data)
            // For now the translation system will skip over these fields, however if there is reason
            // or demand for them to be added back in, let me know.

            if (!isset($result[$model->alias]) || !isset($result[$model->alias]['id'])) {

                continue;
            }

            if (!$is_admin) {

                foreach ($models as $model_name => $model_value) {
                    if (isset($results[$id][$model_name])) {
                        $results[$id][$model_name] = $this->translateArray($results[$id][$model_name], $model_name);
                    }

                }
            } else {

                // Because we're in the admin area, we don't want to translate the field - that'd be a
                // pretty stupid thing to do. What we can however do is append the translation to the model
                // This lets you access the translated value(s). However if we do it the same way we load
                // up the translations (via the configure variable) it'll only be the active language. So
                // instead we'll pull in all the language data.

                if (!isset($all_phrases[$result[$model->alias]['id']])) {
                    continue;
                }
                $phrase = $all_phrases[$result[$model->alias]['id']];

                $params = array(
                    'conditions' => array(
                        'is_active' => '1',
                    )
                );

                // Hide the default language from all translation forms if this is true.
                if (Configure::read('EvTranslation.hide_default_in_admin')) {
                    $params['conditions']['is_default'] = 0;
                }

                $languages = $this->TranslationLanguage->find('all', $params);

                $translated_fields = array();

                $translated_display_fields = array();


                foreach ($phrase['TranslationTranslation'] as $translation) {

                    $translated_fields[$translation['TranslationLanguage']['id']][$phrase['TranslationPhrase']['model_field']] = $translation;
                    $translated_display_fields['lang_' . $translation['TranslationLanguage']['id'] . '_' . $phrase['TranslationPhrase']['model_field']] = $translation['translation'];
                }

                $results[$id][$model->alias]['Translations'] = $translated_fields;
                $results[$id][$model->alias] = array_merge($results[$id][$model->alias], $translated_display_fields);

                // Add empty fields for those languages that don't yet have a translation stored.
                foreach ($languages as $language) {

                    if ($language['TranslationLanguage']['is_default'] == '1') {
                        //continue;
                    }

                    $lang_id = $language['TranslationLanguage']['id'];

                    foreach ($result[$model->alias] as $field => $value) {

                        if (!isset($results[$id][$model->alias]['Translations'][$lang_id][$field]) &&
                            !in_array($field, $this->settings['excluded_fields'])) {

                            // Check field type against the model schema.
                            if (isset($model->_schema[$field]['type'])) {
                                $type = $model->_schema[$field]['type'];

                                if (!in_array($type, $this->ignored_field_types)) {

                                    $results[$id][$model->alias]['Translations'][$lang_id][$field] = array(
                                        'translation' => null,
                                        'TranslationLanguage' => $language['TranslationLanguage']
                                    );

                                    $results[$id][$model->alias]['lang_' . $lang_id . '_' . $field] = null;
                                }
                            }
                        }
                    }
                }
            }
        }

        // Return the modified result set.
        return $results;
    }


    /**
     * After Save
     *
     * Adds or updates the translation phrases and translated data for this model.
     *
     * @param  Model  $model   Model that we're working with
     * @param  boolean $created If true, this is a new record, if false we're updating an existing one.
     * @param  array  $options Options passed from Model::save().
     * @return bool
     */
    public function afterSave(Model $model, $created, $options = array())
    {
        if (!isset($model->data[$model->alias])) {
            return true;
        }

        $model_data = $model->data[$model->alias];

        $this->TranslationPhrase = ClassRegistry::init('TranslationPhrase');
        $this->TranslationTranslation = ClassRegistry::init('TranslationTranslation');

        // Loop the model data so we can get all fields
        foreach ($model_data as $field => $value) {

            // Check each field value to see if its a language string.
            if (strstr($field, 'lang_')) {

                // Reset the models so we don't conflict with any previous runs.
                $this->TranslationPhrase->clear();
                $this->TranslationTranslation->clear();

                // The field is a language translation. Now we need to extract the field name and
                // the language id from the variable. These are stored as lang_<ID>_<FIELD>
                $lang_var = ltrim($field, 'lang_');
                $lang_var = explode("_", $lang_var, 2);

                $language_id = $lang_var[0];
                $lang_field = $lang_var[1];

                // We want to check to see if a translation phrase already exists.
                $phrase = $this->TranslationPhrase->find('first', array(
                    'conditions' => array(
                        'model' => $model->alias,
                        'model_id' => $model->id,
                        'model_field' => $lang_field
                    )
                ));

                // Check to see if the phrase already exists. If it doesnt, we'll create it.
                if (empty($phrase)) {

                    $phrase_data = array(
                        'model' => $model->alias,
                        'model_id' => $model->id,
                        'model_field' => $lang_field,
                        'translation_phrase_group_id' => null,
                        'is_hidden' => '1',
                        'is_active' => '1'
                    );

                    $phrase = $this->TranslationPhrase->save($phrase_data);
                }

                // The language phrase exists. Now check to see if we've already stored a
                // translation. If we have, we'll overwrite it. Otherwise, we'll create it.
                $translation = $this->TranslationTranslation->find('first', array(
                    'conditions' => array(
                        'translation_language_id' => $language_id,
                        'translation_phrase_id' => $phrase['TranslationPhrase']['id']
                    )
                ));

                // Check to see if we've got a translation record
                if (!empty($translation)) {

                    // A record exists! Delete it!
                    $this->TranslationTranslation->delete($translation['TranslationTranslation']['id']);
                }

                // Now create a new record if the value is not empty.
                if ($value !='') {

                    $translation_data = array(
                        'translation_language_id' => $language_id,
                        'translation_phrase_id' => $phrase['TranslationPhrase']['id'],
                        'translation' => $value
                    );

                    // Save it
                    $this->TranslationTranslation->save($translation_data);
                }
            }
        }

        // All done!
        return true;
    }

    /**
     * After Delete
     *
     * Used to remove any orphaned translation records once a model item is deleted.
     *
     * @param  Model  $model Model that we're working with
     * @return bool
     */
    public function afterDelete(Model $model)
    {
        // Load the two translation models we're working with
        $this->TranslationPhrase = ClassRegistry::init('TranslationPhrase');
        $this->TranslationTranslation = ClassRegistry::init('TranslationTranslation');

        // Load up the applicable phrases
        $phrases = $this->TranslationPhrase->find('all', array(
            'conditions' => array(
                'TranslationPhrase.model' => $model->alias,
                'TranslationPhrase.model_id' => $model->id
            )
        ));

        // Loop the phrases so we can get the individual related translation(s) for that phrase.
        foreach ($phrases as $phrase) {

            // Load all translations for this phrase
            $translations = $this->TranslationTranslation->find('all', array(
                'conditions' => array(
                    'TranslationTranslation.translation_phrase_id' => $phrase['TranslationPhrase']['id']
                )
            ));

            // Loop through the translation results
            foreach ($translations as $translation) {

                // Delete the translated phrase
                $this->TranslationTranslation->delete($translation['TranslationTranslation']['id']);
            }

            // Delete the phrase record as it's no longer needed/wanted.
            $this->TranslationPhrase->delete($phrase['TranslationPhrase']['id']);
        }

        // All done!
        return true;
    }

    /**
     * Translate Array
     *
     * Used to replace the values of an array with it's translated values
     *
     * @param  Array $results  Results that we'll be translating
     * @param  Object $model   Model object we'll be translating on
     * @return Array  Returns the modified result set with the translated values
     */
    public function translateArray($results, $model)
    {
        $new_result = array();

        if (is_object($model)) {
            $model_name = $model->alias;
        } else {
            $model_name = $model;
        }

        foreach ($results as $result => $fields) {

            if (is_array($fields)) {
                $sub_model = $model;

                if (!is_int($result)) {
                    $sub_model = $result;
                }

                $new_result[$result] = $this->translateArray($fields, $sub_model);

            } else {

                $field_slug = $model . '_' . $results['id'] . '_' . $result;

                if (isset($this->settings['phrases']) && array_key_exists($field_slug, $this->settings['phrases'])) {

                    // A translation for this field exists. Replace the result value with the
                    // translated one.
                    if ($this->settings['phrases'][$field_slug] !='') {

                        $new_result[$result] = $this->settings['phrases'][$field_slug];
                    } else {
                        $new_result[$result] = $results[$result];
                    }
                } else {
                    $new_result[$result] = $results[$result];
                }

            }
        }

        return $new_result;
    }
}