<?php
/**
 * TranslationUtil
 *
 * This class is used to store the active translations, and provides numerous key functions of the
 * translation plugin
 *
 * @author Rick Mills <rick@evoluted.net>
 */
class TranslationUtil
{
    public static $lang_id = 1; // Default language id
    public static $lang = array(); // Setup empty array for the active language data
    public static $default_lang = array(); // Setup empty array for the default language data

    public static $phrases = array(); // Setup empty array for the language phrases

    public static $TranslationLanguage; // Setup var for where we'll store the translation language model
    public static $TranslationPhrase; // Setup var for where we'll store the translation phrase model

    public static $request; // Stores the request data from the active controller method

    /**
     * Init
     *
     * This is called by the controllers to initiate the translation system.
     *
     * @param  Object $request The current request data from the active controller method
     */
    public static function init($request)
    {
        self::$request = $request; // Populate the gobal request var with the controller request data

        // Loads in the translation models we'll be using
        self::$TranslationLanguage = ClassRegistry::init('EvTranslation.TranslationLanguage');
        self::$TranslationPhrase = ClassRegistry::init('EvTranslation.TranslationPhrase');

        // Set the active language id
        self::$lang_id = Configure::read('EvTranslation.language_id');

        // Load the active language phrases
        self::$lang = self::$TranslationLanguage->loadLanguage(self::$lang_id);

        // Load the default language as a fallback option
        self::$default_lang = self::$TranslationLanguage->loadDefaultLanguage();

        // Write the language phrases into a config variable
        Configure::write('translation.phrases', self::$lang);

        // Check if we're running in the admin area. This is needed for the translatable behavior
        // to work out if it needs to run or not
        if (isset(self::$request->params['admin']) && self::$request->params['admin'] == '1') {
            $is_admin = true;
        } else {
            $is_admin = false;
        }

        // Write the admin status to a configure var just so it's a bit quicker to access elsewhere
        Configure::write('is_admin', $is_admin);
    }

    /**
     * Translate Fields
     *
     * Translates all fields in a data array. This can be uses to set up translations for a model
     * on the fly.
     *
     * @param  array  $data  Model array of all fields that need to be translated
     * @return array  Returns the modified data array containing the translated data fields
     */
    public static function translateFields($data)
    {
        // Loop through the main data array
        foreach ($data as $key => $value) {

            // If the value contains an array it means it's got the model fields inside it.
            if (is_array($value)) {

                // Loop through the model fields
                foreach ($value as $model_field => $model_value) {

                    // If the value is not an array, it means we've found the model value.
                    if (!is_array($model_value)) {

                        // Set up the vars we'll need to build the updated field language data.
                        $modelName = $key;
                        $modelId = $data[$key]['id'];
                        $modelField = $model_field;

                        if (isset(self::$lang[$modelName.'_'.$data[$key]['id'].'_'.$model_field]) && self::$lang[$modelName.'_'.$data[$key]['id'].'_'.$model_field] !='') {
                            $data[$key][$model_field] = self::$lang[$modelName.'_'.$data[$key]['id'].'_'.$model_field];
                        }
                    }
                    // TODO: Add support for sub-array translations.
                }
            }
        }
        return $data;
    }

    /**
     * Translated Field
     *
     * Gets the translated field based on provided variables.
     *
     * @param  String  $model_name  The model name
     * @param  Integer  $id  The model record id
     * @param  String  $field_name  Name of the model field
     * @return Mixed  Returns either the language value, or false if no value is found
     */
    public static function translateField($model_name, $id, $field_name)
    {
        if (isset(self::$lang[$model_name.'_'.$id.'_'.$field_name]) && self::$lang[$model_name.'_'.$id.'_'.$field_name] != '') {
            return self::$lang[$model_name.'_'.$id.'_'.$field_name];
        }

        return false;
    }

    /**
     * Translate By Slug
     *
     * Pulls an individual translation record based on a provided slug name.
     *
     * @param  String  $slug  Unique slug of the translation field
     * @return  String  Returns the translated slug if one was found
     */
    public static function translateBySlug($slug)
    {
        return self::$lang[$slug];
    }

    /**
     * Admin Form Fields Translated
     *
     * This method creates the translation fields for the admin area editor.
     *
     * @param  Object $request  The request object from the controller
     * @param  array  $fields  The field array that'll be supplied by the controlers _adminFormFields method
     * @return Returns the array of translation fields
     */
    public static function _adminFormFieldsTranslated($request, $fields = array())
    {
        $translations = array();

        // Yo dawg! I heard you like loops, so I looped the loop with loops!

        // Loop through all models
        foreach ($request->data as $model => $model_fields) {

            // If the model fields var exists and isnt empty, continue.
            if (is_array($model_fields) && !empty($model_fields)) {
                // Loop through all model fields (because we need to find a sub-array of translations)
                foreach ($model_fields as $field => $value) {

                    // Only run the next bit if the Translations field is found.
                    if ($field == 'Translations') {

                        // Loop through the translations
                        foreach ($value as $language_id => $field_translation) {

                            // If the field translation var is an array, continue.
                            if (is_array($field_translation)) {

                                // Loop through each of the field translation records, pulling out the
                                // field nanme and the actual translation data
                                foreach ($field_translation as $field_name => $translation) {

                                    // Only run this next bit if we've not yet created the basic
                                    // translation data array for this language, and only if the
                                    // translation actually has a TranslationLanguage param set.
                                    if (!isset($translations[$language_id]) && isset($translation['TranslationLanguage'])) {

                                        // Create the empty translation array for this language and
                                        // set the label up that'll be shown on the admin area
                                        // fieldset legend above the translatable fields.
                                        $translations[$language_id] = array(
                                            'fields' => array(),
                                            'label' => $translation['TranslationLanguage']['language']
                                        );

                                    }

                                    // If the model has a record for this field, add a translatable
                                    // field for this record.
                                    if (isset($fields[$model.'.'.$field_name])) {
                                        $translations[$language_id]['fields'][$model.'.lang_' . $language_id . '_' . $field_name] = $fields[$model.'.'.$field_name];
                                    }

                                    // Set the field label as otherwise it'll look like a complete mess.
                                    // We're setting it to {Field Name} ({Language}).
                                    if (isset($translation['TranslationLanguage'])) {
                                        $translation_field_name = InflectorExt::humanize($field_name);
                                        $translations[$language_id]['fields'][$model.'.lang_' . $language_id . '_' . $field_name]['label'] = $translation_field_name;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        // If we've got no translations found, it basically means that there aren't any records in the
        // translation phrases table yet. It does not mean that the model doesnt need/support translations.
        //
        // The below will basically loop through the main model fields and add the translatable fields array.
        // We skip over any records that are listed in the models excluded fields array, as well as any
        // non-translatable fields such as integers, maps and dates.
        if (empty($translations)) {

            // The translation phrases has no record of any translations ever being done for this table.
            // Because of this we need to load the translatable fields in manually.
            $excluded_types = array(
                'integer',
                'boolean',
                'datetime',
                'numeric',
                'select',
                'multiselect'
            );

            // Load up the active, non-default languages then loop through them.
            $languages = self::$TranslationLanguage->find('all', array(
                'conditions' => array(
                    'is_active' => '1',
                    'is_default' => '0'
                )
            ));

            // Loop the languages!
            foreach ($languages as $language) {

                // Set up the array for this languages translations
                $translations[$language['TranslationLanguage']['id']] = array(
                    'label' => $language['TranslationLanguage']['language'],
                    'fields' => array()
                );

                // Loop trough the model data so we can build up the translatable fields.
                foreach ($request->data as $model => $model_fields) {

                    // Load the model - we need this to pull the excluded fields list (if one exists)
                    $this_model = ClassRegistry::init($model);

                    // Check for any excluded fields so we can skip over those.
                    // I'm well aware that this is a horrible way of doing this, so thank's for
                    // offering to improve it!
                    $excluded_fields = array();

                    if (isset($this_model->actsAs['EvTranslation.Translatable']['excluded_fields'])) {

                        $excluded_fields = $this_model->actsAs['EvTranslation.Translatable']['excluded_fields'];

                    }

                    // Loop through each model field so we can use it's field type and label and then add
                    // them to the individual language subarray.
                    if (is_array($model_fields) && ! empty($model_fields)) {

                        foreach ($model_fields as $field => $value) {

                            if (isset($fields[$model.'.'.$field]) && !in_array($field, $excluded_fields) && !in_array($fields[$model.'.'.$field]['type'], $excluded_types)) {

                                $translations[$language['TranslationLanguage']['id']]['fields'][$model.'.lang_' . $language['TranslationLanguage']['id'] . '_' . $field] = $fields[$model.'.'.$field];

                                $translation_field_name = InflectorExt::humanize($field);
                                $translations[$language['TranslationLanguage']['id']]['fields'][$model.'.lang_' . $language['TranslationLanguage']['id'] . '_' . $field]['label'] = $translation_field_name;
                            }
                        }
                    }
                }
            }
        }

        // By this point, if we're supporting translations, we'll have the full array of translatable
        // fields, all ready to be used.
        return $translations;
    }

    /**
     * Lang
     *
     * Used to load up a specific language slug, and falls back to the default language value should
     * a translated value not be found. This is primarily used inside views to translate custom slugs.
     *
     * @param  String $slug The uniqye slug we'll be requesting a translation record for.
     * @return String Returns either the translation record, the default record, or as a last resort, the slug.
     */
    public static function lang($slug)
    {
        if (isset(self::$lang[$slug])) {

            return self::$lang[$slug];

        } elseif (isset(self::$default_lang[$slug])) {
            // The slug wasn't found in the active translations, so try loading it from the default
            // language.
            return self::$default_lang[$slug];

        } else {

            // TODO: Add an additional fallback to try and show the original model value instead of
            // the language slug. The downside with this is that it'll create extra queries, something
            // I'm trying to avoid.

            // Return the orginal slug as no translation was found.
            return $slug;
        }
    }
}