<?php

App::uses('EvTransactionsAppModel', 'EvTransactions.Model');

class Transaction extends EvTransactionsAppModel {

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'id';

	public $hasMany = array(
		'TransactionsItem' => array(
			'className' => 'EvTransactions.TransactionsItem'
		),
		'TransactionData' => array(
			'className' => 'EvTransactions.TransactionData'
		)
	);

	public $belongsTo = array(
		'Currency' => array(
			'className' => 'EvCurrency.Currency'
		)
	);

	/**
	 * redefiune the model constructor so we can setup the dynamic relationship
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		// setup dynamic relation
		$relationships = Configure::read('EvTransactions.transactionRelation');
		if (! empty($relationships)) {
			$belongsTo = array();
			foreach ($relationships as $key => $value) {
				$belongsTo[$key] = $value;
			}

			if (! empty($belongsTo)) {
				$this->bindModel(
					array(
						'belongsTo' => $belongsTo
					),
					false
				);
			}
		}
	}

	/**
	 * redefine to add in contains
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForView($id, $params = array()) {
		$params['contain'][] = 'TransactionsItem';
		$params['contain'][] = 'TransactionData';

		return parent::readForView($id, $params);
	}

	/**
	 * afterFind to transform the transactionData
	 * into easier to use format
	 *
	 * @param mixed $results The results of the find operation
	 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
	 * @return mixed Result of the find operation
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#afterfind
	 */
	public function afterFind($results, $primary = false) {
		if (isset($results['0']['TransactionData'])) {
			foreach ($results as $key => $data) {
				$results[$key]['TransactionData'] = Hash::combine(
					$data['TransactionData'],
					'{n}.name',
					'{n}.data'
				);
			}
		}

		return $results;
	}

	/**
	 * afterSave to catch any errors from save
	 * @param  boolean $created true if new record or false if update
	 * @param  array  $options options passed through to save
	 */
	public function afterSave($created, $options = array()) {
		if (!empty($this->data['Transaction']['status'])) {
			if ($this->data['Transaction']['status'] == 'failed') {
				if (CakePlugin::loaded('EvErrbit')) {
					//If a message exists then use that otherwise use generic error message.
					if (isset($this->data['Transaction']['message']) && !empty($this->data['Transaction']['message'])) {
						$e = new Exception($this->data['Transaction']['message']);
					} else {
						$e = new Exception('Transaction failed');
					}
					Errbit::notify($e);
				}
			}
		}

		return true;
	}

	/**
	 * return the array to use for retrieving transaction listing
	 *
	 * @param int user_id we are pulling transactions for
	 * @param int|false int to limit number of rows or false to return all rows
	 * @return array array of data for paginate function / model->find
	 */
	public function transactionListParams($userId, $limit = false) {
		$contain = Configure::read('EvTransactions.listing.contain');
		$contain = array_merge(array('TransactionsItem'), $contain);

		$return = array(
			'contain' => $contain,
			'conditions' => array(
				'Transaction.user_id' => $userId,
				'Transaction.status' => 'success'
			),
			'order' => 'Transaction.id DESC'
		);

		if ($limit !== false) {
			$return['limit'] = $limit;
		}

		return $return;
	}

	/**
	 * given the transaction ID and the payment token
	 * check the details match and return valid transaction
	 *
	 * @param 	int 	$transactionId 	The transaction row id
	 * @param 	string 	$paymentToken 	The payment token
	 * @return 	bool
	 */
	public function validTransaction($transactionId, $paymentToken) {
		$count = $this->find(
			'count',
			array(
				'conditions' => array(
					'Transaction.id' => $transactionId,
					'Transaction.payment_token' => $paymentToken
				),
				'callbacks' => false
			)
		);

		return (bool)$count;
	}

	/**
	 * take the key => value array of data
	 * and format for transactionData model
	 *
	 * @param 	array 	$data 			Transaction data to be formatted
	 * @return 	array 	$formattedData 	Formatted transactionData
	 */
	public function buildData($data) {
		$formattedData = array();

		foreach ($data as $name => $data) {
			$formattedData[] = array(
				'name' => $name,
				'data' => $data
			);
		}

		return $formattedData;
	}
}
