<?php

App::uses('CustomEmail', 'Lib');
App::uses('AppComponent', 'Controller/Component');

class TransactionsComponent extends AppComponent {

	protected $_gateway = null;

	public $id = null;

	const FREE_PAYMENT_GATEWAY = 'freePayment';

	/**
	 * force the page onto SSL if it isn't already
	 */
	public function forceSSL() {
		if (! $this->_controller->request->is('ssl')) {
			return $this->_controller->redirect('https://' . env('SERVER_NAME') . $this->here);
		}
	}

	/**
	 * take and process a payment
	 *
	 * based on the config gateway settings, take and process a payment from the gateway given the details
	 *
	 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
	 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
	 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
	 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount', optional elements are model and model_id
	 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
	 * @param array $transactionData - array of key => value data to store on the transaction for use later (e.g. address ID numbers for delivery)
	 * @param int 	$userId - user id of the user to attach the transaction to, default will be logged in user.
	 * @param string $gateway - a gateway override, will default to main Transaction.gateway set in config
	 * @return mixed  - return from the getPayment gateway function. Depends on what and whether that gateway needs anything returning
	 */
	public function takePayment($return, $model, $amount, $items, $extra = null, $transactionData = array(), $userId = null, $gateway = null) {
		if (is_array($amount)) {
			$total = $amount['amount'];
			$currency = $amount['currency'];
		} else {
			$total = $amount;
			$currency = CakeSession::read('EvCurrency.currencyCode');
		}

		$gateway = $this->_selectPaymentGateway(
			$model,
			$total,
			$items,
			$extra,
			$transactionData,
			$userId,
			$gateway
		);

		// load the component
		$this->_gateway = $this->loadGateway($gateway);
		$this->_gateway->setup();

		// setup the transactions model
		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		if (empty($userId)) {
			$user = $this->_controller->Auth->user();
			$userId = null;

			if (isset($user['User']['id'])) {

				$userId = $user['User']['id'];
			} elseif (isset($user['id'])) {

				$userId = $user['id'];
			}
		}

		$data = array(
			'Transaction' => array(
				'id' => !empty($extra['transaction_id']) ? $extra['transaction_id'] : null,
				'user_id' => $userId,
				'payment_method' => $gateway,
				'transaction_amount' => $total,
				'currency' => $currency,
				'currency_id' => CakeSession::read('EvCurrency.currencyId'),
				'status' => 'initiated',
				'model' => (isset($model['model']) && ! empty($model['model'])) ? $model['model'] : null,
				'model_id' => (isset($model['model_id']) && ! empty($model['model_id'])) ? $model['model_id'] : null
			),
			'TransactionsItem' => array()
		);

		foreach ($items as $item) {
			$data['TransactionsItem'][] = array(
				'description' => $item['description'],
				'amount' => $item['amount'],
				'model' => (isset($item['model'])) ? $item['model'] : null,
				'model_id' => (isset($item['model_id'])) ? $item['model_id'] : null,
			);
		}

		if (! empty($transactionData)) {
			$data['TransactionData'] = $this->Transaction->buildData($transactionData);
		}

		$this->Transaction->saveAll(
			$data,
			array(
				'deep' => true
			)
		);

		$this->id = $this->Transaction->id;

		$this->_gateway->setupPayment($this->Transaction->id, $return, $model, $amount, $items, $extra);

		if (
			(Configure::read('EvTransactions.skipGatewayOnLocal') && getEnvironment() == 'local') ||
			(Configure::read('EvTransactions.skipGatewayOnDev') && getEnvironment() !== 'production')
		) {
			// Dev mode active, skip payment gateway and mark as paid.
			$this->Transaction->save(array(
				'status' => 'success',
				'message' => 'DEV MODE - Fake payment.'
			));

		} else {
			$result = $this->_gateway->getPayment($this->Transaction->id);
		}

		return $this->_controller->redirect($return['return'] . '?transaction=' . $this->Transaction->id);
	}

/**
 * Select the payment gateway to process the transaction based on the current transaction.
 *
 * @param array $model   Array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items.
 * @param float $total   The total price of the transaction.
 * @param array $items   The items in the transaction.
 * @param mixed $extra   Extra data to pass to the gateway.
 * @param array $data    The transaction data of the current transaction.
 * @param int   $userId  The id of the user performing the transaciton.
 * @param string $gateway The currently selected gateway.
 * @return string The gateway to use when processing a transaction.
 */
	protected function _selectPaymentGateway(
		$model,
		$total,
		$items,
		$extra,
		$data,
		$userId,
		$gateway
	) {
		//Check price of payment, if 0 we want to use the free payment gateway
		if ($total == 0) {
			return 'EvTransactions.FreePayment';
		}

		//If the gateway has been provided then use that.
		if (!empty($gateway)) {
			return $gateway;
		}

		//Fallback on the plugin config.
		return Configure::read('EvTransactions.gateway');
	}

	/**
	 * checkPayment
	 *
	 * load the gateway we are dealing with then check the return data
	 *
	 * @param string $gateway - a gateway override, will default to main Transaction.gateway set in config
	 * @return bool - true or false as to whether it was sucessful
	 */
	public function checkPayment($gateway = null) {
		// setup the transactions model
		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		if (isset($this->_controller->request->query['transaction'])) {
			//Get the gateway from the passed transaction
			$transaction = $this->Transaction->findById($this->_controller->request->query['transaction']);

			if ($transaction['Transaction']['transaction_amount'] == 0) {
				$gateway = 'EvTransactions.FreePayment';
			} elseif (
				(Configure::read('EvTransactions.skipGatewayOnLocal') && getEnvironment() == 'local') ||
				(Configure::read('EvTransactions.skipGatewayOnDev') && getEnvironment() !== 'production')
			) {
				// Dev mode active, skip payment gateway and mark as paid.
				return true;
			} elseif (empty($gateway)) {
				// find the gateway from the transaction, the config or return false if neither are set
				if (!empty($transaction['Transaction']['payment_method'])) {
					$gateway = $transaction['Transaction']['payment_method'];
				} elseif (Configure::check('EvTransactions.gateway')) {
					$gateway = Configure::read('EvTransactions.gateway');
				} else {
					return false;
				}
			}
		}

		// load the component
		$this->_gateway = $this->loadGateway($gateway);
		$this->_gateway->setup();

		// process the actual return values in the gateway
		$results = $this->_gateway->processReturn();

		// update the transaction status
		if (isset($results['transaction_id']) && ! empty($results['transaction_id'])) {

			$this->id = $this->Transaction->id = $results['transaction_id'];
			$transaction = array(
				'Transaction' => array(
					'status' => (isset($results['result']) && $results['result'] === true) ? 'success' : 'failed',
				)
			);

			// only update the message if we have one
			if (!empty($results['message'])) {
				$transaction['Transaction']['message'] = $results['message'];
			}

			$this->Transaction->save($transaction);
		}

		return (isset($results['result'])) ? $results['result'] : false;
	}

/**
 * Authorise payment in the gateway
 * Stripe needs this for 2FA
 *
 * @param array $params a list of parameters
 * @param string $gateway - a gateway override, will default to main gateway set in config
 * @return mixed
 */
	public function authorisePayment(array $params, $gateway = null) {
		if (empty($gateway)) {
			$gateway = Configure::read('EvTransactions.gateway');
		}
		$this->_gateway = $this->loadGateway($gateway);
		$this->_gateway->setup();

		return $this->_gateway->authorisePayment($params);
	}

	/**
	 * send a receipt to the given email address
	 *
	 * @param array two main elements of 'to' / 'bcc' passed. Sub arrays of 'email' or 'email' => 'name' format.
	 * @param string subject to use on the emails
	 * @param array main elements allowed:
	 * - 'date' - display date for receipt
	 * - 'header' - allows you to pass stuff to receipt template above the table
	 * - 'footer' - allows you to pass stuff to receipt template below the table
	 * - 'overview' - receipt overview
	 * - 'grandtotal' - Grand total
	 * - 'currencyId' - ID of the currency in use
	 * - 'items' - sub array containing the receipt lines
	 * - - 'description' - line item description
	 * - - 'total' - line item total
	 * - - 'tax' - line item tax
	 * - - 'subtotal' - line item total minus tax
	 * @param string template override for main email template
	 * @param string template override for main email layout
	 */
	public function sendReceipt($emails, $subject, $data, $template = 'receipt', $layout = 'receipt') {
		$CustomEmail = new CustomEmail();

		$CustomEmail->template('EvTransactions.' . $template, 'EvTransactions.' . $layout);
		$CustomEmail->theme('Site');
		$CustomEmail->from(array(Configure::read('SiteSetting.general.admin_email') => Configure::read('SiteSetting.general.site_title')));
		$CustomEmail->subject($subject);
		$CustomEmail->viewVars(array(
			'data' => $data
		));
		$CustomEmail->helpers(array(
			'Html'
		));

		if (isset($emails['to']) && ! empty($emails['to'])) {
			foreach ($emails['to'] as $email => $name) {
				if (is_numeric($email)) {
					$email = $name;
					$name = null;
				}

				$CustomEmail->addTo($email, $name);
			}
		}

		if (isset($emails['bcc']) && ! empty($emails['bcc'])) {
			foreach ($emails['bcc'] as $email => $name) {
				if (is_numeric($email)) {
					$email = $name;
					$name = null;
				}

				$CustomEmail->addBcc($email, $name);
			}
		}

		$CustomEmail->send();
	}

	/**
	 * update the transaction record and mark it as failed withj the given message
	 *
	 * @param 	int 	$transactionId 	 	The transaction ID number
	 * @param 	string 	$message 			The failed message
	 * @return mixed On success Model::$data if its not empty or true, false on failure
	 */
	public function failTransaction($transactionId, $message) {
		$Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		$Transaction->id = $transactionId;
		return $Transaction->save(
			array(
				'status' => 'failed',
				'message' => $message
			)
		);
	}

	/**
	 * load the gateway component
	 *
	 * @param 	string 	$gateway 		The gateway to load
	 * @return 	object 	$GatewayObject 	The gateway object
	 */
	public function loadGateway($gateway) {
		return $this->_controller->loadComponent($gateway);
	}
}
