<?php

App::uses('Component', 'Controller');
App::import('Lib', 'Transactions.GatewayInterface');
App::import('Vendor', 'Transactions.Stripe', array(
	'file' => 'Stripe' . DS . 'Stripe.php'
));

class StripeComponent extends Component implements GatewayInterface
{
	private $__controller = null;

	protected $_api = null;

	protected $_config = null;

	private $__charge = null;

	/**
	 * Sets a reference to the calling controller from within the component.
	 *
	 * @see Component::initialize()
	 */
	public function initialize(Controller $controller) {

		parent::initialize($controller);

		$this->__controller = $controller;
	}

	/**
	 * init
	 *
	 * Use the function setup and connection / settings that need setting up in the gateway
	 */
	public function setup()
	{
		if (Configure::read('db.config') == 'live') {

			$this->_config = Configure::read('Transactions.stripe.live');
		} else {

			$this->_config = Configure::read('Transactions.stripe.dev');
		}

		Stripe::setApiKey($this->_config['secret_key']);
	}

	/**
	 * setupPayment
	 *
	 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
	 *
	 * @param int - transaction id we have created
	 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
	 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
	 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
	 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
	 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
	 */
	public function setupPayment($transaction_id, $return, $model, $amount, $items, $extra = array())
	{
		if (is_array($amount)) {

			$amount = (isset($amount['amount'])) ? $amount['amount'] : 0;
			$currency_code = (isset($amount['currency'])) ? $amount['currency'] : Configure::read('Transactions.currency');
		} else {

			$currency_code = Configure::read('Transactions.currency');
		}

		$this->__charge = array(
			'amount' => ($amount * 100),
			'currency' => $currency_code,
			'card' => (isset($this->__controller->request->data['stripeToken']) && ! empty($this->__controller->request->data['stripeToken'])) ? $this->__controller->request->data['stripeToken'] : null,
			'description' => (isset($extra['charge_description']) ? $extra['charge_description'] : ''),
			'metadata' => array(
				'transaction_id' => $transaction_id
			)
		);
	}

	/**
	 * getPayment
	 *
	 * Everything should be setup, actually take the payment
	 * @param int transactions id
	 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
	 */
	/**
	 * NOTE: MOST GATEWAYS WOULD BE AT THIS POINT SENDING YOU OFF TO THEIR SITE FOR PAYMENT
	 * AND THEN REDIRECTING BACK TO A DIFFERENT PAGE FOR THE FUNCTION AT THE BOTTOM (PROCESSRETURN) TO BE RUN.
	 * BECAUSE OF THE WAY STRIPE WORKS IT'S EASIER TO PROCESS HERE, THE VALUE OF THIS FUNCTION IS RETURNED TO THE CONTROLLER
	 * VIA THE TRANSACTIONS->TAKEPAYMENT() FUNCTION SO THERE ISN'T AN ISSUE CHECKING SUCCESS / FAILURE STATES
	 */
	public function getPayment($transaction_id)
	{
		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transaction_id;


		try {

			$stripe_charge = Stripe_Charge::create($this->__charge);

			if (is_object($stripe_charge)) {

				// load the transactions model and update with the token
				$this->Transaction->save(array(
					'Transaction' => array(
						'payment_token' => $stripe_charge->id,
						'status' => 'success'
					)
				));
				$result = true;

			} else {

				$this->Transaction->save(array(
					'Transaction' => array(
						'status' => 'failed',
						'message' => 'There was an unknown error'
					)
				));
				$result = false;
			}

		} catch(Exception $e) {

			// load the transactions model and update with the token
			$this->Transaction->save(array(
				'Transaction' => array(
					'status' => 'failed',
					'message' => $e->getMessage()
				)
			));
			$result = false;
		}

		return array(
			'result' => $result,
			'transaction_id' => $transaction_id
		);
	}

	/**
	 * return
	 *
	 * deal with a return from the gateway and check for success / fail
	 *
	 * @return array - with three elements,
	 *				 - 'result' = true/false value
	 *  			 - 'message' = text message about transaction (i.e. reason for failing)
	 * 				 - 'transaction_id' = int of the transaction row
	 */
	public function processReturn() { }

}
