<?php

App::uses('Component', 'Controller');
App::import('Lib', 'Transactions.GatewayInterface');

class StripeCheckoutComponent extends Component implements GatewayInterface {

	public $Transaction;

	protected $_config = [];

	protected $_charge = [];

	protected $_locales = [
		'dan' => 'da',
		'deu' => 'de',
		'eng' => 'en',
		'spa' => 'es',
		'fin' => 'fi',
		'fre' => 'fr',
		'ita' => 'it',
		'jpn' => 'ja',
		'nor' => 'nb',
		'nld' => 'nl',
		'pol' => 'pl',
		'por' => 'pt',
		'swe' => 'sv',
		'zho' => 'zh',
	];

/**
 * {@inheritDoc}
 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);
		$this->__controller = $controller;
	}

/**
 * {@inheritDoc}
 */
	public function setup() {
		$this->_config = Configure::read('Transactions.stripe.live');
		if (Configure::read('db.config') !== 'live') {
			$this->_config = Configure::read('Transactions.stripe.dev');
		}

		\Stripe\Stripe::setApiKey($this->_config['secret_key']);
	}

/**
 * {@inheritDoc}
 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = []) {
		$currencyCode = Configure::read('Transactions.currency');

		if (is_array($amount)) {
			$amount = (isset($amount['amount'])) ? $amount['amount'] : 0;
			$currencyCode = (isset($amount['currency'])) ? $amount['currency'] : $currencyCode;
		}

		$this->_charge = [
			'customer_email' => $extra['user']['email'],
			'payment_method_types' => ['card'],
			'line_items' => [
				[
					'name' => 'Total',
					'amount' => $amount * 100,
					'currency' => $currencyCode,
					'quantity' => 1,
				],
			],
			'success_url' => $return['return'],
			'cancel_url' => $return['cancel'],
			'locale' => $this->_getLocale(! empty($extra['language']) ? $extra['language'] : null),
		];
	}

/**
 * {@inheritDoc}
 */
	public function getPayment($transactionId) {
		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transactionId;

		$paymentSuccess = false;

		try {
			$stripeCharge = \Stripe\Checkout\Session::create($this->_charge);

			if (is_object($stripeCharge) && $stripeCharge->id) {
				$this->Transaction->save([
					'payment_token' => $stripeCharge->id,
					'status' => 'initiated',
				]);
				$this->__controller->Session->write('Transactions.stripe_checkout.payment_token', $stripeCharge->id);
				$paymentSuccess = true;
			} else {
				$this->Transaction->save([
					'status' => 'failed',
					'message' => 'There was an unknown error',
				]);
			}

		} catch (\Exception $exception) {
			$this->Transaction->save([
				'status' => 'failed',
				'message' => $exception->getMessage(),
			]);
		}

		return [
			'result' => $paymentSuccess,
			'transaction_id' => $transactionId,
		];
	}

/**
 * {@inheritDoc}
 *
 * @throws BadRequestException
 */
	public function processReturn() {
		$events = \Stripe\Event::all([
			'type' => 'checkout.session.completed',
			'created' => [
				'gte' => time() - 60 * 60,
			],
		]);

		$paymentToken = $this->__controller->Session->consume('Transactions.stripe_checkout.payment_token');

		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$transaction = $this->Transaction->findByPaymentToken($paymentToken);

		if (empty($transaction)) {
			throw new BadRequestException('Transaction not found');
		}

		// Seek out the event associated with our payment.
		$event = null;
		foreach ($events->data as $e) {
			if ($e->data->object->id === $paymentToken) {
				$event = $e;
				break;
			}
		}

		if ($event !== null && $event->type === 'checkout.session.completed') {
			$transaction['status'] = 'success';
			$this->Transaction->save($transaction);

			return [
				'result' => true,
				'message' => 'Payment successful',
				'transaction_id' => $transaction['Transaction']['id'],
			];
		}

		return [
			'result' => false,
			'message' => 'Payment was unsuccessful',
			'transaction_id' => $transaction['Transaction']['id'],
		];
	}

/**
 * Get locale
 *
 * @param string $language Language
 * @return string
 */
	protected function _getLocale($language = null) {
		return ! empty($this->_locales[$language]) ? $this->_locales[$language] : 'auto';
	}
}
