<?php
App::uses('Component', 'Controller');
App::import('Lib', 'Transactions.GatewayInterface');

use Omnipay\Omnipay as Omnipay;

class EpayComponent extends Component implements GatewayInterface {

/**
 * Define an array of language code mappings.
 *
 * @var array
 * @see http://tech.epay.dk/en/payment-window-parameters
 */
	protected $_languages = array(
		'dan' => 1,
		'eng' => 2,
		'swe' => 3,
		'nor' => 4,
		'kal' => 5,
		'ice' => 6,
		'deu' => 7,
		'fin' => 8,
		'spa' => 9,
		'fre' => 10,
		'pol' => 11,
		'ita' => 12,
		'dut' => 13
	);

	public function initialize(Controller $controller) {
		parent::initialize($controller);
		$this->__controller = $controller;

		$this->_api = Omnipay::create('Epay');
	}

/**
 * Use the function setup and connection / settings that need setting up in the gateway
 */
	public function setup() {
		$prodGateway = Configure::read('Transactions.epay.live');
		$stagingGateway = Configure::read('Transactions.epay.dev');

		if (CakePlugin::loaded('BuzzSites')) {
			$Site = ClassRegistry::init('BuzzSites.Site');
			$site = $Site->getActiveSite();

			if (
				isset($site['Site']['production_gateway_credentials']) &&
				isset($site['Site']['staging_gateway_credentials'])
			) {
				$prodGateway = json_decode($site['Site']['production_gateway_credentials'], true);
				$stagingGateway = json_decode($site['Site']['staging_gateway_credentials'], true);
			}
		}

		if (Configure::read('db.config') === 'live' || Configure::read('transactions.forceLive')) {
			$this->_config = $prodGateway;
		} else {
			$this->_config = $stagingGateway;
			$this->_api->setTestMode(true);
		}

		$this->_api->setMerchantnumber($this->_config['merchant_number']);
		$this->_api->setSecret($this->_config['secret']);
		$this->_api->setWindowstate(3);
		$this->_api->setTimeout(!empty($this->_config['timeout']) ? $this->_config['timeout'] : 15);
	}

/**
 * Setup payment.
 *
 * @param int $transactionId transaction id we have created
 * @param array $return array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
 * @param array $model array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
 * @param float|array $amount amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
 * @param array $items multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
 * @param mixed $extra variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = []) {
		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transactionId;

		$description = 'Payment to ' . $this->_config['brandname'];
		// Override generic description with custom one if exists
		if (isset($extra['description'])) {
			$description = $extra['description'];
		}

		// Set the language code.
		$defaultLanguage = !empty($this->_config['language']) ? $this->_config['language'] : 'eng';
		$language = !empty($extra['language']) ? $extra['language'] : $defaultLanguage;
		$language = !empty($this->_languages[$language]) ? $this->_languages[$language] : $this->_languages[$defaultLanguage];

		// Set currency
		if (is_array($amount)) {
			$amount = $amount['amount'];
			$currency = $amount['currency'];
		} else {
			$currency = Configure::read('Transactions.currency');
		}

		$this->_data = [
			'amount' => $amount * 100,
			'currency' => $currency,
			'description' => $description,
			'callbackurl' => $return['return'],
			'accepturl' => $return['return'],
			'cancelurl' => $return['cancel'],
			'language' => $language,
			'orderid' => $transactionId
		];

		$this->__controller->Session->write('Transactions.epay.data', $this->_data);
	}

/**
 * Get payment
 *
 * Everything should be setup, actually take the payment
 *
 * @param int $transactionId Transactions id
 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
 */
	public function getPayment($transactionId) {
		$response = $this->_api->purchase($this->_data)->send();

		return $response->redirect();
	}

/**
 * return
 *
 * deal with a return from the gateway and check for success / fail
 *
 * @return array - with three elements,
 *               - 'result' = true/false value
 *               - 'message' = text message about transaction (i.e. reason for failing)
 *               - 'transaction_id' = int of the transaction row
 */
	public function processReturn() {
		// Read the session data and then destroy it.
		$this->_data = $this->__controller->Session->read('Transactions.epay.data');
		$this->__controller->Session->delete('Transactions.epay.data');
		// Complete the purchase.
		$response = $this->_api->completePurchase()->send();

		// Save the ePay response.
		$data = [
			'transaction_id' => $this->_data['orderid'],
			'txnid' => $response->getTransactionReference(),
			'query' => $_SERVER['QUERY_STRING']
		];
		ClassRegistry::init('Transactions.Epay')->save($data);

		return [
			'result' => $response->isSuccessful(),
			'message' => $response->isSuccessful() ? 'Payment successful' : 'Payment was unsuccessful',
			'transaction_id' => $this->_data['orderid']
		];
	}

}
