<?php
App::uses('Component', 'Controller');
App::import('Lib', 'Transactions.GatewayInterface');

use GuzzleHttp\Client;

class BamboraComponent extends Component implements GatewayInterface {

	protected $_apiKey = null;

	protected $_controller = null;

/**
 * Language code mappings.
 * @var array
 */
	protected $_languages = [
		'eng' => 'en-GB',
		'swe' => 'sv-SE',
	];

/**
 * Initialize component
 *
 * @param Controller $controller Controller
 * @return void
 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);
		$this->_controller = $controller;
	}

/**
 * Set up
 *
 * @return void
 */
	public function setup() {
		if (Configure::read('db.config') === 'live' || Configure::read('transactions.forceLive')) {
			$this->_config = Configure::read('Transactions.bambora.live');
		} else {
			$this->_config = Configure::read('Transactions.bambora.dev');
		}
		$this->_apiKey = $this->_generateApiKey(
			$this->_config['access'],
			$this->_config['merchant_number'],
			$this->_config['secret']
		);
	}

/**
 * Generate the API key
 *
 * @param string $accessToken    Access token
 * @param string $merchantNumber Merchant number
 * @param string $secretToken    Secret token
 * @return string API key
 */
	protected function _generateApiKey($accessToken, $merchantNumber, $secretToken) {
		return base64_encode($accessToken . "@" . $merchantNumber . ":" . $secretToken);
	}

/**
 * Set up the payment
 *
 * @param int $transactionId Transaction ID
 * @param array $return Return and cancel urls
 * @param array $model Model and model ID
 * @param float|array $amount Amount due
 * @param array $items Transaction items
 * @param array $extra Extra parameters
 * @return void
 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = []) {
		// Set currency
		if (is_array($amount)) {
			$amount = $amount['amount'];
			$currency = $amount['currency'];
		} else {
			$currency = Configure::read('Transactions.currency');
		}

		$siteId = (! empty($extra['api_site_id']) ? $extra['api_site_id'] : (! empty($extra['site_id']) ? $extra['site_id'] : 0));

		$amount = $amount * 100; // Convert to minor units for bambora.
		$this->_data = [
			'instantcaptureamount' => $amount,
			'order' => [
				'id' => $transactionId,
				'amount' => $amount,
				'currency' => $currency,
				'site_id' => $siteId
			],
			'url' => [
				'accept' => $return['return'],
				'cancel' => $return['cancel'],
				'callbacks' => [
					['url' => $return['return']]
				],
				'immediateredirecttoaccept' => 1,
			],
			'paymentwindow' => [
				'id' => 1,
				'paymentmethods' => [
					// Disable swish payments
					[
						'id' => 'swish',
						'action' => 'exclude'
					],
				]
			],
			'site_id' => $siteId
		];

		if (! empty($extra['language']) && ! empty($this->_languages[$extra['language']])) {
			$this->_data['paymentwindow']['language'] = $this->_languages[$extra['language']];
		}
		$this->_controller->Session->write('Transactions.bambora.data', $this->_data);
	}

/**
 * Get payment
 *
 * @param int $transactionId Transaction ID
 * @return mixed
 */
	public function getPayment($transactionId) {
		$checkoutUrl = "https://api.v1.checkout.bambora.com/sessions";
		$requestJson = json_encode($this->_data);
		$contentLength = isset($requestJson) ? strlen($requestJson) : 0;
		$headers = array(
			'Content-Type: application/json',
			'Content-Length: ' . $contentLength,
			'Accept: application/json',
			'Authorization: Basic ' . $this->_apiKey
		);

		// We write these out before doing the browser redirect as we cant pass any
		// of this data via bambora to get to the return url. So by doing this we
		// can pick the data up based on the transaction id on the return. This allows
		// our multisite setup to work with two bambora accounts.
		CakeSession::write('transaction_' . $transactionId . '_config', $this->_config);
		CakeSession::write('transaction_' . $transactionId . '_apiKey', $this->_apiKey);

		$curl = curl_init();
		curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "POST");
		curl_setopt($curl, CURLOPT_POSTFIELDS, $requestJson);
		curl_setopt($curl, CURLOPT_URL, $checkoutUrl);
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
		curl_setopt($curl, CURLOPT_FAILONERROR, false);
		curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2);

		$rawResponse = curl_exec($curl);
		$response = json_decode($rawResponse);

		return $this->_controller->redirect($response->url);
	}

/**
 * Process return from payment gateway
 *
 * @return array Return result
 */
	public function processReturn() {
		$this->_data = $this->_controller->Session->consume('Transactions.bambora.data');

		// Required for multisite compatability as this gateway doesn't provide a way for us to pass data through their
		// redirect, so we have a pre-created session with the order (transaction) id for the config and api data.
		$this->_config = $this->_controller->Session->consume('transaction_' . $this->_data['order']['id'] . '_config');
		$this->_apiKey = $this->_controller->Session->consume('transaction_' . $this->_data['order']['id'] . '_apiKey');

		$transactionParams = $this->_controller->params['url'];
		$result = false;
		if (! empty($transactionParams['hash'])) {
			$hash = $transactionParams['hash'];
			unset($transactionParams['hash']);
			$result = md5(implode($transactionParams) . $this->_config['md5']) === $hash;
			ClassRegistry::init('Transactions.Bambora')->save([
				'transaction_id' => $this->_data['order']['id'],
				'txnid' => $transactionParams['txnid'],
				'query' => $_SERVER['QUERY_STRING'],
				'site_id' => $this->_data['site_id']
			]);
		}

		return [
			'result' => $result,
			'message' => $result ? 'Payment successful' : 'Payment was unsuccessful. ' .  print_r($transactionParams, true) . ' /// ' . print_r($this->_data, true),
			'transaction_id' => $this->_data['order']['id'],
		];
	}
}
