<?php

App::uses('Component', 'Controller');
App::import('Lib', 'Transactions.GatewayInterface');

use Omnipay\Omnipay as Omnipay;

class PayPalComponent extends Component implements GatewayInterface {

    private $__controller = null;

    protected $_config = null;

    protected $_api = null;

    protected $_data = null;

    public function initialize(Controller $controller) {
        parent::initialize($controller);

        $this->__controller = $controller;

        $this->_api = Omnipay::create('PayPal_Express');
    }

    /**
     * init
     *
     * Use the function setup and connection / settings that need setting up in the gateway
     */
    public function setup() {
        if (Configure::read('db.config') == 'live') {
            $this->_config = Configure::read('Transactions.paypal.live');
        } else {
            $this->_config = Configure::read('Transactions.paypal.dev');
        }

        $this->_api->setUsername($this->_config['username']);
        $this->_api->setPassword($this->_config['password']);
        $this->_api->setSignature($this->_config['signature']);
        $this->_api->setBrandName($this->_config['brandname']);

        if ($this->_config['imageurl'] != null) {
            $this->_api->setHeaderImageUrl($this->_config['imageurl']);
        }

        $this->_api->setTestMode($this->_config['testmode']);
    }

    /**
     * setupPayment
     *
     * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
     *
     * @param int - transaction id we have created
     * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
     * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
     * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
     * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
     * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
     */
    public function setupPayment($transaction_id, $return, $model, $amount, $items, $extra = array()) {

        $this->Transaction = ClassRegistry::init('Transactions.Transaction');
        $this->Transaction->id = $transaction_id;

        $description = 'Payment to ' . $this->_config['brandname'];
        // Override generic description with custom one if exists
        if (isset($extra['description'])) {
            $description = $extra['description'];
        }

        //Set currency
        if (is_array($amount)) {
            $amount = $amount['amount'];
            $currency = $amount['currency'];
        } 
        else {
            $currency = Configure::read('Transactions.currency');
        }

        $this->_data = array(
            'amount' => $amount,
            'description' => $description,
            'returnUrl' => $return['return'],
            'cancelUrl' => $return['cancel'],
            'currency' => $currency,
            'transaction_id' => $transaction_id
        );
		
		$noShipping = Configure::read('Transactions.paypal.noShipping');
		
		if($noShipping !== NULL) {
			$this->_data['noShipping'] = $noShipping;
		}

        $this->__controller->Session->write('Transactions.paypal.data', $this->_data);
    }

    /**
     * getPayment
     *
     * Everything should be setup, actually take the payment
     *
     * @param int transactions id
     * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
     */
    public function getPayment($transaction_id) {

        $response = $this->_api->purchase($this->_data)->send();

        if ($response->isRedirect()) {

            // omnipay wants to redirect
            $response->redirect();

        }
        else {
            // no redirect, check if successfull
            if ($response->isSuccessful()) {
                $redirect_url = $this->_data['returnUrl'];
            }
        }

        // check if redirect has been set (was successful) if not redirect to cancel
        if (! isset($redirect_url)) {
            $redirect_url = $this->_data['cancelUrl'];
        }

        header("Location:" . $redirect_url);

        // Ensure no further code gets excuted as we are redirecting
        exit;
    }

    /**
     * return
     *
     * deal with a return from the gateway and check for success / fail
     *
     * @return array - with three elements,
     *               - 'result' = true/false value
     *               - 'message' = text message about transaction (i.e. reason for failing)
     *               - 'transaction_id' = int of the transaction row
     */
    public function processReturn() {
        // Read the session data
        $this->_data = $this->__controller->Session->read('Transactions.paypal.data');

        // Remove the session data (good practice)
        $this->__controller->Session->delete('Transactions.paypal.data');

        // Send to complete the purchase
        $response = $this->_api->completePurchase($this->_data)->send();

        // check response
        if ($response->isSuccessful()) {
            return array(
                    'result'    =>  true,
                    'message'   =>  'Payment successful',
                    'transaction_id'    =>  $this->_data['transaction_id']
                );
        } 
        else {
            return array(
                    'result'    =>  false,
                    'message'   =>  'Payment was unsuccessful',
                    'transaction_id'    =>  $this->_data['transaction_id']
                );
        }
    }

}