<?php

App::uses('Component', 'Controller');
App::import('Lib', 'Transactions.GatewayInterface');
App::import('Vendor', 'Transactions.SagepayServerApi');

class SagepayServerComponent extends Component implements GatewayInterface
{
	private $__controller = null;

	protected $_api = null;

	protected $_data = null;

	protected $_error_url = null;


/**
 * Sets a reference to the calling controller from within the component.
 *
 * @see Component::initialize()
 */
	public function initialize(Controller $controller) {

		parent::initialize($controller);

		$this->__controller = $controller;

		$this->_api = new SagePayServerApi();
	}


/**
 * init
 *
 * Use the function setup and connection / settings that need setting up in the gateway
 */
	public function setup() {

		if (Configure::read('db.config') == 'live') {

			$this->_config = Configure::read('Transactions.sagepay_server.live');
		} else {

			$this->_config = Configure::read('Transactions.sagepay_server.dev');
			$this->_api->setTestMode(true);
		}

		$this->_api->setVendor($this->_config['vendor']);
	}


/**
 * setupPayment
 *
 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
 *
 * @param int - transaction id we have created
 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
 * @return void
 */
	public function setupPayment($transaction_id, $return, $model, $amount, $items, $extra=array()) {

		$vendorTxCode = $this->_api->makeVendorTxCode($transaction_id);
		$this->_api->setVendorTxCode($vendorTxCode);

		$description = empty($extra['description']) ? 'Payment to '.Configure::read('SiteSetting.site_title') : $extra['description'];
		$this->_api->setDescription($description);

		if (is_array($amount)) {
			$amount = $amount['amount'];
			$currency_code = $amount['currency'];
		} else {
			$currency_code = Configure::read('Transactions.currency');			
		}

		$this->_api->setAmount(number_format($amount, 2), $currency_code);

		$data = array(
			'BillingFirstnames' => $extra['user_details']['firstname'],
			'BillingSurname' => $extra['user_details']['lastname'],
			'CustomerEMail' => $extra['user_details']['email'],
			'BillingAddress1' => $extra['billing_address']['address1'],
			'BillingAddress2' => $extra['billing_address']['address2'],
			'BillingCity' => $extra['billing_address']['city'],
			'BillingPostCode' => $extra['billing_address']['postcode'],
			'BillingCountry' => $extra['billing_address']['country_code'],
			'DeliveryFirstnames' => $extra['user_details']['firstname'],
			'DeliverySurname' => $extra['user_details']['lastname'],
			'DeliveryAddress1' => $extra['billing_address']['address1'],
			'DeliveryAddress2' => $extra['billing_address']['address2'],
			'DeliveryCity' => $extra['billing_address']['city'],
			'DeliveryPostCode' => $extra['billing_address']['postcode'],
			'DeliveryCountry' => $extra['billing_address']['country_code']
		);

		if (isset($return['return']) && ! empty($return['return'])) {

			$data['NotificationUrl'] = $return['return'];
		}

		if (isset($return['error']) && ! empty($return['error'])) {

			$this->_error_url = $return['error'];
		}

		$this->_data = $data;

		return;
	}


/**
 * getPayment
 *
 * Everything should be setup, actually take the payment
 *
 * @param int transactions id
 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
 */
	public function getPayment($transaction_id) {
		$response = $this->_api->makePayment($this->_data);

		if (isset($response['Status']) && $response['Status'] == 'OK') {

			$this->SagepayServer = ClassRegistry::init('Transactions.SagepayServer');

			$date = date('Y-m-d H:i:s');
			$this->SagepayServer->save(array(
				'SagepayServer' => array(
					'VPSTxId' => $response['VPSTxId'],
					'SecurityKey' => $response['SecurityKey'],
					'VendorTxCode' => $this->_api->getVendorTxCode(),
					'transaction_id' => $transaction_id,
					'created' => $date,
					'modified' => $date
				)
			));

			$this->__controller->redirect($response['NextURL']);

		} else {

			$this->__controller->redirect($this->_error_url);
		}
	}


/**
 * return
 *
 * deal with a return from the gateway and check for success / fail
 *
 * @return array - with three elements,
 *				 - 'result' = true/false value
 *  			 - 'message' = text message about transaction (i.e. reason for failing)
 * 				 - 'transaction_id' = int of the transaction row
 */
	public function processReturn() {

		$response['VPSTxId'] = isset($this->__controller->request->data['VPSTxId']) ? $this->__controller->request->data['VPSTxId'] : '';
		$response['vendorTxCode'] = isset($this->__controller->request->data['VendorTxCode']) ? $this->__controller->request->data['VendorTxCode'] : '';
		$response['status'] = isset($this->__controller->request->data['Status']) ? $this->__controller->request->data['Status'] : '';
		$response['txAuthNo'] = isset($this->__controller->request->data['TxAuthNo']) ? $this->__controller->request->data['TxAuthNo'] : '';
		$response['avscv2'] = isset($this->__controller->request->data['AVSCV2']) ? $this->__controller->request->data['AVSCV2'] : '';
		$response['addressResult'] = isset($this->__controller->request->data['AddressResult']) ? $this->__controller->request->data['AddressResult'] : '';
		$response['postcodeResult'] = isset($this->__controller->request->data['PostCodeResult']) ? $this->__controller->request->data['PostCodeResult'] : '';
		$response['cv2Result'] = isset($this->__controller->request->data['CV2Result']) ? $this->__controller->request->data['CV2Result'] : '';
		$response['giftAid'] = isset($this->__controller->request->data['GiftAid']) ? $this->__controller->request->data['GiftAid'] : '';
		$response['3dSecureStatus'] = isset($this->__controller->request->data['3DSecureStatus']) ? $this->__controller->request->data['3DSecureStatus'] : '';
		$response['cavv'] = isset($this->__controller->request->data['CAVV']) ? $this->__controller->request->data['CAVV'] : '';
		$response['addressStatus'] = isset($this->__controller->request->data['AddressStatus']) ? $this->__controller->request->data['AddressStatus'] : '';
		$response['payerStatus'] = isset($this->__controller->request->data['PayerStatus']) ? $this->__controller->request->data['PayerStatus'] : '';
		$response['cardType'] = isset($this->__controller->request->data['CardType']) ? $this->__controller->request->data['CardType'] : '';
		$response['last4Digits'] = isset($this->__controller->request->data['Last4Digits']) ? $this->__controller->request->data['Last4Digits'] : '';
		$response['StatusDetail'] = isset($this->__controller->request->data['StatusDetail']) ? $this->__controller->request->data['StatusDetail'] : '';

		$this->SagepayServer = ClassRegistry::init('Transactions.SagepayServer');
		$payment = $this->SagepayServer->find('first', array(
			'conditions' => array('VPSTxId' => $response['VPSTxId']),
		));

		if (
			(isset($payment['SagepayServer']['SecurityKey']) && ! empty($payment['SagepayServer']['SecurityKey'])) &&
			(isset($payment['SagepayServer']['transaction_id']) || ! empty($payment['SagepayServer']['transaction_id']))
		) {
			$signature = $response['VPSTxId'] . $response['vendorTxCode'] . $response['status'] . $response['txAuthNo']
				. $this->_config['vendor'] . $response['avscv2'] . $payment['SagepayServer']['SecurityKey']
				. $response['addressResult'] . $response['postcodeResult'] . $response['cv2Result'] . $response['giftAid']
				. $response['3dSecureStatus'] . $response['cavv'] . $response['addressStatus'] . $response['payerStatus']
				. $response['cardType'] . $response['last4Digits'];

			$signature = strtoupper(md5($signature));

			$this->Transaction = ClassRegistry::init('Transactions.Transaction');
			$this->Transaction->id = $payment['SagepayServer']['transaction_id'];
			$this->Transaction->save(array(
				'Transaction' => array(
					'payment_token' => $response['VPSTxId']
				)
			));

			if (! isset($this->__controller->request->data['VPSSignature']) || $signature !== $this->__controller->request->data['VPSSignature']) {

				return array(
					'result' => false,
					'message' => 'Cannot match the MD5 Hash. Order might be tampered with.',
					'transaction_id' => $payment['SagepayServer']['transaction_id']
				);

			} elseif ($this->__controller->request->data['Status']=='ABORT') {

				return array(
					'result' => false,
					'message' => 'Order was aborted.',
					'transaction_id' => $payment['SagepayServer']['transaction_id']
				);
			} else {

				if (isset($this->__controller->request->data['Status']) && $this->__controller->request->data['Status'] == 'OK') {

					return array(
						'result' => true,
						'message' => (! empty($response['StatusDetail'])) ? $response['StatusDetail'] : '',
						'transaction_id' => $payment['SagepayServer']['transaction_id']
					);
				} else {

					return array(
						'result' => false,
						'message' => (! empty($response['StatusDetail'])) ? $response['StatusDetail'] : '',
						'transaction_id' => $payment['SagepayServer']['transaction_id']
					);
				}
			}
		} else {

			return array(
				'result' => false,
				'message' => 'There was an error checking the MD5 hash, the order might have been tampered with'
			);
		}
	}


	public function postBack($redirectUrl) {

		ob_start();
		header("Content-type: text/plain");

		$this->Transaction = ClassRegistry::init('Transactions.Transaction');

		$txId = $response['VPSTxId'] = isset($this->__controller->request->data['VPSTxId']) ? $this->__controller->request->data['VPSTxId'] : '';

		$message = $this->Transaction->field('message', array(
			'Transaction.payment_token' => $txId
		));

		$output = '';

		if ($message == 'Cannot match the MD5 Hash. Order might be tampered with.') {

			$redirectUrl .= '/0';

			$output .= "Status=INVALID\nRedirectURL=$redirectUrl\n";

			$output .= "StatusDetail=" . $message . "\n";

		} elseif ($message == 'Order was aborted.') {

			$redirectUrl .= '/2';

			$output .= "Status=ABORT\n";

			$output .= "RedirectURL=$redirectUrl\n";

		} else {

			if (isset($this->__controller->request->data['Status']) && ! empty($this->__controller->request->data['Status'])) {

				$redirectUrl .= $this->__controller->request->data['Status'] != 'OK' ? "/0" : "/1";

				$output .= "Status=".($this->__controller->request->data['Status'] != 'OK' ? "INVALID" : "OK")."\n";
			} else {

				$redirectUrl .= '/0';
				$output .= "Status=INVALID\n";
			}

			$output .= "RedirectURL=$redirectUrl\n";
		}

		ob_end_clean();

		echo $output;
	}

}
