<?php

App::uses('Component', 'Controller');
App::import('Lib', 'Transactions.GatewayInterface');
App::import('Vendor', 'Transactions.SagepayServerApi');
use Omnipay\Omnipay as Omnipay;

class SagepayDirectComponent extends Component implements GatewayInterface
{
	private $__controller = null;

	protected $_api = null;

	protected $_data = null;

	protected $_error_url = null;

	/**
	 * Sets a reference to the calling controller from within the component.
	 *
	 * @see Component::initialize()
	 */
	public function initialize(Controller $controller) {

		parent::initialize($controller);

		$this->__controller = $controller;

		$this->_api = Omnipay::create('SagePay_Direct');

	}

	/**
	 * init
	 *
	 * Use the function setup and connection / settings that need setting up in the gateway
	 */
	public function setup() {

		if (Configure::read('db.config') == 'live') {

			$this->_config = Configure::read('Transactions.sagepay_direct.live');

		} else {

			$this->_config = Configure::read('Transactions.sagepay_direct.dev');

			$this->_api->setTestMode(true);

			if (Configure::read('db.config') == 'local') {

				$this->_api->setSimulatorMode(true);
			}
		}

		$this->_api->setVendor($this->_config['vendor']);

	}

	/**
	 * setupPayment
	 *
	 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
	 *
	 * @param int - transaction id we have created
	 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
	 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
	 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
	 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
	 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
	 */
	public function setupPayment($transaction_id, $return, $model, $amount, $items, $extra=array()) {

		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transaction_id;

		$state = '';
		if (isset($extra['address']['country']) && strtolower($extra['address']['country']) == 'us' && ! empty($extra['address']['county'])) {

			$state = $extra['address']['county'];
		}

		$cardData = array(
			'firstName' => $extra['card']['first_name'],
			'lastName' => $extra['card']['last_name'],
			'number' => $extra['card']['card_number'],
			'startMonth' => (!empty($extra['card']['start_month'])) ? $extra['card']['start_month'] : null,
			'startYear' => (!empty($extra['card']['start_year'])) ? $extra['card']['start_year'] : null,
			'expiryMonth' => $extra['card']['expiry_month'],
			'expiryYear' => $extra['card']['expiry_year'],
			'cvv' => $extra['card']['security_number'],
			'address1' => $extra['address']['address_line_1'],
			'address2' => $extra['address']['address_line_2'],
			'city' => $extra['address']['city'],
			'postcode' => $extra['address']['postcode'],
			'state' => $state,
			'country' => (! empty($extra['address']['country'])) ? $extra['address']['country'] : 'GB',
			'email' => $extra['user']['email'],
			'phone' => ''
		);

		//Set currency
		if (is_array($amount)) {

			$amount = $amount['amount'];
			$currency_code = $amount['currency'];

		} else {

			$currency_code = Configure::read('Transactions.currency');

		}


		$params = array(
			'transactionId' => $transaction_id."-".time(),
			'transactionReference' => 'test',
			'returnUrl' => $return['return'],
			'currency' => $currency_code,
			'amount' => $amount,
			'description' => 'Payment to '.Configure::read('SiteSetting.site_title'),
			'clientIp' => $this->__controller->request->clientIp()
		);

		$this->__controller->Session->write('Transactions.SagepayDirect.params', $params);

		$params['card'] = $cardData;

		$this->_data = $this->_api->purchase(
			$params
		);

	}


	/**
	 * getPayment
	 *
	 * Everything should be setup, actually take the payment
	 *
	 * @param int transactions id
	 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
	 */
	public function getPayment($transaction_id) {

		$result = $this->_sendRequest($this->_data);

		return array(
			'result' => $result,
			'transaction_id' => $transaction_id
		);

	}

	/**
	 * return
	 *
	 * deal with a return from the gateway and check for success / fail
	 *
	 * @return array - with three elements,
	 *				 - 'result' = true/false value
	 *  			 - 'message' = text message about transaction (i.e. reason for failing)
	 * 				 - 'transaction_id' = int of the transaction row
	 */
	public function processReturn() { return $this->process3dSecure(); }


	public function process3dSecure() {

		$params = $this->__controller->Session->read('Transactions.SagepayDirect.params');

		$request = $this->_api->completePurchase($params);

		$result = $this->_sendRequest($request);

		return array(
			'result' => $result,
			'transaction_id' => $params['transactionId']
		);


	}


	protected function _sendRequest($request) {

		$params = $this->__controller->Session->read('Transactions.SagepayDirect.params');

		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = substr($params['transactionId'], 0, strpos($params['transactionId'], '-'));

		try {

			$response = $request->send();

			//Customer is successfully paid
			if($response->isSuccessful()) {

				// load the transactions model and update with the token
				$this->Transaction->save(array(
					'Transaction' => array(
						'payment_token' => $response->getTransactionReference(),
						'status' => 'success'
					)
				));
				$result = true;

			} elseif ($response->isRedirect()) {

				$response->redirect(); // this will automatically forward the customer. For 3D secure etc.

			} else {

				$this->Transaction->save(array(
					'Transaction' => array(
						'status' => 'failed',
						'message' => $response->getMessage()
					)
				));
				$result = false;
			}

		} catch(Exception $e) {

			// load the transactions model and update with the token
			$this->Transaction->save(array(
				'Transaction' => array(
					'status' => 'failed',
					'message' => $e->getMessage()
				)
			));
			$result = false;
		}

		return $result;

	}

}
