<?php

App::uses('AppBehavior', 'Model/Behavior');

class TemplateBehavior extends AppBehavior {

	/**
	 * Setup this behavior with the specified configuration settings.
	 *
	 * @param Model $model Model using this behavior
	 * @param array $config Configuration settings for $model
	 * @return void
	 */
	public function setup(Model $Model, $config = array()) {
		$Model->addBehaviour('EvCustomFields.CustomFieldsData', array('formInject' => true));
	}

	/**
	 * get the image / document blocks for the selected template
	 * then associate the model wityh the image / docs so we can save
	 *
	 * @param Model $model Model using this behavior
	 * @param array $options Options passed from Model::save().
	 * @return mixed False if the operation should abort. Any other result will continue.
	 * @see Model::save()
	 */
	public function afterValidate(Model $Model) {
		$return = parent::afterValidate($Model);

		// when calling from within behavior we need to manually add the Model var in.
		if (isset($Model->data[$Model->alias]['template_id']) && ! empty($Model->data[$Model->alias]['template_id'])) {
			$this->setupAttachmentRelationships($Model, $Model->data[$Model->alias]['template_id']);
		}

		return $return;
	}

	/**
	 * beforeBeforeSave callback,
	 * run before saveAll is called, allowing you to modify the whole data array
	 * before save processes
	 *
	 * @param 	object 		Model - populated by Cakes ORM
	 * @param 	array 		Array to save
	 * @param 	array 		Options array
	 * @return 	array|bool 	Bool false to stop save, return modified data array to proceed
	 */
	public function beforeBeforeSave($Model, $data, $options = array()) {
		if (empty($data[$Model->alias]['template_id'])) {
			return $data;
		}

		foreach ($data as $key => $item) {
			if (! is_array($item)) {
				continue;
			}

			$attachmentType = Hash::extract($item, '{n}.attachment_type');
			if (empty($attachmentType)) {
				continue;
			}

			$data[$key] = Hash::insert($item, '{n}.template_id', $data[$Model->alias]['template_id']);
		}

		return $data;
	}

	/**
	 * beforeSave start processing if we need to delete custom fields / images / docs
	 *
	 */
	public function beforeSave(Model $Model, $options = array()) {
		$return = parent::beforeSave($Model, $options);

		// if another callback already returned false, just stop now
		if (! $return) {
			return false;
		}

		// check for id, if no idea must be new return true
		if (
			empty($Model->data[$Model->alias][$Model->primaryKey])
			|| empty($Model->data[$Model->alias]['template_id'])
		) {
			return true;
		}

		// get the old template id so we can see if they have selected a new template
		$oldTemplateId = $Model->field(
			'template_id',
			array(
				$Model->alias . '.' . $Model->primaryKey => $Model->data[$Model->alias][$Model->primaryKey]
			)
		);

		// the intvals don't match. process the old CF data, images and documents
		if (intval($oldTemplateId) !== intval($Model->data[$Model->alias]['template_id'])) {
			$FieldData = EvClassRegistry::init('EvCustomFields.FieldData');
			$Image = EvClassRegistry::init('EvCore.Image');
			$Document = EvClassRegistry::init('EvCore.Document');

			$ImageBlock = EvClassRegistry::init('EvTemplates.ImageBlock');
			$DocumentBlock = EvClassRegistry::init('EvTemplates.DocumentBlock');
			$oldImageBlocks = $this->getAllImageBlocks($Model, $ImageBlock, $oldTemplateId);
			$oldDocBlocks = $this->getAllDocumentBlocks($Model, $DocumentBlock, $oldTemplateId);

			$Model->toDelete = array();
			$Model->toDelete['customFields'] = $FieldData->find(
				'all',
				array(
					'conditions' => array(
						'FieldData.model' => EvClassRegistry::getNameFromModel($Model),
						'FieldData.model_id' => $Model->data[$Model->alias][$Model->primaryKey]
					),
					'callbacks' => false
				)
			);
			$Model->toDelete['customFields'] = Hash::extract($Model->toDelete['customFields'], '{n}.FieldData.id');

			$Model->toDelete['images'] = $Image->find(
				'all',
				array(
					'conditions' => array(
						'Image.model' => $Model->alias,
						'Image.model_id' => $Model->data[$Model->alias][$Model->primaryKey],
						'Image.attachment_type' => $oldImageBlocks,
						'Image.template_id' => $oldTemplateId
					),
					'callbacks' => false
				)
			);

			$Model->toDelete['images'] = Hash::extract($Model->toDelete['images'], '{n}.Image.id');

			$Model->toDelete['documents'] = $Document->find(
				'all',
				array(
					'conditions' => array(
						'Document.model' => $Model->alias,
						'Document.model_id' => $Model->data[$Model->alias][$Model->primaryKey],
						'Document.attachment_type' => $oldDocBlocks
					),
					'callbacks' => false
				)
			);
			$Model->toDelete['documents'] = Hash::extract($Model->toDelete['documents'], '{n}.Document.id');
		}

		return $return;
	}

	/**
	 * get all the image blocks - dynamic / static
	 *
	 * @param 	Model 	$Model 			The model this behaviour is attached to
	 * @param 	Model 	$ImageBlock 	Instance of EvTemplates.ImageBlock
	 * @param 	int 	$templateId 	The template id
	 * @return 	array 	$allBlocks 	The found image blocks
	 */
	public function getAllImageBlocks($Model, $ImageBlock, $templateId) {
		$allBlocks = $ImageBlock->getBlocksForAttachmentType($templateId);

		if (is_array($Model->imageSlots)) {
			$blockNames = array_keys($Model->imageSlots);
			foreach ($blockNames as $key => $name) {
				$name = Inflector::camelize($name);
				if ($name == 'Main') {
					$allBlocks[] = 'Image';
				} else {
					$allBlocks[] = $name . 'Image';
				}
			}
		} else {
			$allBlocks[] = 'Image';
		}

		return $allBlocks;
	}

	/**
	 * get all the document blocks - dynamic / static
	 *
	 * @param 	Model 	$Model 			The model this behaviour is attached to
	 * @param 	Model 	$DocumentBlock 	Instance of EvTemplates.DocumentBlock
	 * @param 	int 	$templateId 	The template id
	 * @return 	array 	$allBlocks 		The found documentat blocks
	 */
	public function getAllDocumentBlocks($Model, $DocumentBlock, $templateId) {
		$oldDocBlocks = $DocumentBlock->getBlocksForAttachmentType($templateId);

		if (is_array($Model->imageSlots)) {
			$blockNames = array_keys($Model->imageSlots);
			foreach ($blockNames as $key => $name) {
				$name = Inflector::camelize($name);
				if ($name == 'Main') {
					$oldDocBlocks[] = 'Document';
				} else {
					$oldDocBlocks[] = $name . 'Document';
				}
			}
		} else {
			$oldDocBlocks[] = 'Document';
		}

		return $oldDocBlocks;
	}

	/**
	 * afterSave, process any custom fields, images, documents that were marked for deletion
	 * within the beforeSave callback
	 *
	 * @param Model $model Model using this behavior
	 * @param bool $created True if this save created a new record
	 * @param array $options Options passed from Model::save().
	 * @return bool
	 * @see Model::save()
	 */
	public function afterSave(Model $Model, $created, $options = array()) {
		parent::afterSave($Model, $created, $options);

		if (! empty($Model->toDelete['customFields'])) {
			$FieldData = EvClassRegistry::init('EvCustomFields.FieldData');

			$FieldData->deleteAll(
				array(
					'FieldData.id' => $Model->toDelete['customFields']
				)
			);
		}

		if (! empty($Model->toDelete['images'])) {
			$Image = EvClassRegistry::init('EvCore.Image');

			$Image->deleteAll(
				array(
					'Image.id' => $Model->toDelete['images']
				),
				true,
				true
			);
		}

		if (! empty($Model->toDelete['documents'])) {
			$Document = EvClassRegistry::init('EvCore.Document');

			$Document->deleteAll(
				array(
					'Document.id' => $Model->toDelete['documents']
				),
				true,
				true
			);
		}

		return true;
	}

	/**
	 * given the template id setup the image / docs relationships
	 *
	 * @param   Model $model Model using this behavior
	 * @param   int     templateId
	 * @return  void
	 */
	public function setupAttachmentRelationships($Model, $templateId) {
		$imgBlocks = $this->getAttachmentBlocks('EvTemplates.ImageBlock', $templateId);
		$docBlocks = $this->getAttachmentBlocks('EvTemplates.DocumentBlock', $templateId);

		$this->setupBlock($Model, 'imageSlots', 'ImageBlock', $imgBlocks, $templateId);
		$this->setupBlock($Model, 'documentSlots', 'DocumentBlock', $docBlocks, $templateId);

		$Model->initialiseAssociateModels();
	}

	/**
	 * setup a single attachment block
	 *
	 * @param   Model $model Model using this behavior
	 * @param   string  Type of block imageSlots|documentSlots
	 * @param   string  Model Alias ImageBlock|DocumentBlock
	 * @param   array   Array of blocks
	 * @return  void
	 */
	public function setupBlock($Model, $type, $modelAlias, $blocks, $templateId = null) {
		if (! empty($blocks)) {
			$slots = array();

			foreach ($blocks as $block) {
				$conditions = array();

				if (! empty($templateId)) {
					$prefix = Inflector::camelize($block[$modelAlias]['name']) . str_replace('Block', '', $modelAlias);
					$conditions[$prefix . '.template_id'] = $templateId;
				}

				$slots[$block[$modelAlias]['name']] = array(
					'slots' => $block[$modelAlias]['slots'],
					'fields' => $block[$modelAlias]['blockFields'],
					'conditions' => $conditions
				);

				if (isset($block[$modelAlias]['dimensions'])) {
					$slots[$block[$modelAlias]['name']]['dimensions'] = $block[$modelAlias]['dimensions'];
				}
			}
			$Model->$type = array_merge(
				(is_array($Model->$type)) ? $Model->$type : array(),
				$slots
			);
		}
	}

	/**
	 * get image and document blocks
	 *
	 * @param   string  Attachment type
	 * @param   int     templateId
	 * @return  array|   array of image blocks / document blocks
	 */
	public function getAttachmentBlocks($attachmentType, $templateId) {
		$attachmentModel = EvClassRegistry::init($attachmentType);
		$blocks = $attachmentModel->find(
			'all',
			array(
				'conditions' => array(
					$attachmentModel->alias . '.is_active' => 1,
					$attachmentModel->alias . '.template_id' => $templateId
				)
			)
		);

		if (! empty($blocks)) {
			return $blocks;
		} else {
			return false;
		}
	}

	/**
	 * get images for an item retrieved not via readForEdits
	 *
	 * @param   object          Model Object - Cake handles this
	 * @param   array           array of item data
	 * @param   string|array    image block(s) to retrieve
	 * @param   string          model alias
	 * @return  array
	 */
	public function getImages($Model, $data, $blocks, $modelAlias = null) {
		if (empty($modelAlias)) {
			$modelAlias = $Model->alias;
		}

		if (! isset($data[$modelAlias]['id']) || empty($data[$modelAlias]['id'])) {
			return array();
		}

		if (! is_array($blocks)) {
			$blocks = array($blocks);
		}

		$Image = EvClassRegistry::init('EvCore.Image');

		$images = $Image->find(
			'all',
			array(
				'conditions' => array(
					'Image.model' => $modelAlias,
					'Image.model_id' => $data[$modelAlias]['id'],
					'Image.attachment_type' => $blocks
				),
				'order' => 'Image.sequence ASC'
			)
		);

		if (empty($images)) {
			return array();
		}

		return Hash::combine($images, '{n}.Image.id', '{n}.Image', '{n}.Image.attachment_type');
	}

	/**
	 * get documents for an item retrieved not via readForEdits
	 *
	 * @param   object          Model Object - Cake handles this
	 * @param   array           array of item data
	 * @param   string|array    document block(s) to retrieve
	 * @param   string          model alias
	 * @return  array
	 */
	public function getDocuments($Model, $data, $blocks, $modelAlias = null) {
		if (empty($modelAlias)) {
			$modelAlias = $Model->alias;
		}

		if (! isset($data[$modelAlias]['id']) || empty($data[$modelAlias]['id'])) {
			return array();
		}

		if (! is_array($blocks)) {
			$blocks = array($blocks);
		}

		$Document = EvClassRegistry::init('EvCore.Document');

		$documents = $Document->find(
			'all',
			array(
				'conditions' => array(
					'Document.model' => $modelAlias,
					'Document.model_id' => $data[$modelAlias]['id'],
					'Document.attachment_type' => $blocks
				),
				'order' => 'Document.sequence ASC'
			)
		);

		if (empty($documents)) {
			return array();
		}

		return Hash::combine($documents, '{n}.Document.id', '{n}.Document', '{n}.Document.attachment_type');
	}
}
