<?php

App::uses('EvTemplatesAppModel', 'EvTemplates.Model');
App::uses('InflectorExt', 'EvInflector.Lib');

abstract class AttachmentBlock extends EvTemplatesAppModel {

	public $belongsTo = array(
		'Template' => array(
			'className' => 'EvTemplates.Template'
		)
	);

	/**
	 * get all the blocks for the given ID and return in image / document attachment_type format
	 *
	 * @param 	int 	Template Id to retrieve blocks for
	 * @return 	array 	Array of all the matching blocks with either Image or Document appended
	 */
	public function getBlocksForAttachmentType($templateId) {
		$blocks = $this->find(
			'list',
			array(
				'conditions' => array(
					$this->alias . '.template_id' => $templateId
				)
			),
			array(
				'callbacks' => false
			)
		);

		// walk over all and append the alias type
		array_walk(
			$blocks,
			function (&$value, $key) {
				$value .= str_replace('Block', '', $this->alias);
			}
		);

		return $blocks;
	}

	/**
	 * redefine the beforeSave to add checks for the name. Default to main if none set
	 * Also serialize the blockFields
	 */
	public function beforeSave($options = array()) {
		$return = parent::beforeSave($options);

		if (! empty($this->data[$this->alias])) {
			// default any blocks with no name to "Main"
			if (empty($this->data[$this->alias]['name'])) {
				$this->data[$this->alias]['name'] = 'main';
			}

			// camelCase any names so we can get MainImage / ListingImage / GalleryThumbImage
			$this->data[$this->alias]['name'] = InflectorExt::camelize($this->data[$this->alias]['name']);

			// Serialise any select fields
			$this->data[$this->alias]['blockFields'] = json_encode($this->data[$this->alias]['blockFields']);
		}

		return $return;
	}

	/**
	 * redefine the afterFind to unserialize any fields
	 *
	 */
	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		if (! empty($results)) {
			foreach ($results as $key => $item) {
				if (! empty($item[$this->alias]['blockFields'])) {
					$results[$key][$this->alias]['blockFields'] = json_decode($item[$this->alias]['blockFields'], true);
				}
			}
		}

		return $results;
	}

	/**
	 * redefine the after delete to delete any actual images / documents that may exist
	 *
	 */
	public function beforeDelete($cascade = true) {
		parent::beforeDelete($cascade);

		$this->data = $this->findById($this->id);

		return true;
	}

	public function afterDelete() {
		parent::afterDelete();

		$class = str_replace('Block', '', get_class($this));

		$Model = EvClassRegistry::init('EvCore.' . $class);
		$Model->deleteAll(
			array(
				$class . '.template_id' => $this->data[$this->alias]['template_id'],
				$class . '.attachment_type' => $this->data[$this->alias]['name'] . $class
			),
			true,
			true
		);
	}

}
