<?php

App::uses('EvTemplatesAppController', 'EvTemplates.Controller');
App::uses('Folder', 'Utility');
App::uses('TemplateUtil', 'Lib');

class TemplatesController extends EvTemplatesAppController {

	public function __construct($request = null, $response = null) {
		parent::__construct($request, $response);

		$this->adminActions[] = 'admin_tpl_preview_img';
	}

	/**
	 * Prepare data for populating admin dropdowns.
	 */
	protected function _adminPopulateLookups() {
		parent::_adminPopulateLookups();

		$templates = $this->_getSelectableTemplates();
		$attachmentBlockFields = $this->_getBlockFields();
		$helperFunctions = $this->_getHelperFunctions();

		$this->set(compact('templates', 'attachmentBlockFields', 'helperFunctions'));
	}

	/**
	 * get the list of available methods on the Templates Utility
	 *
	 * @return array 	array of available methods
	 */
	protected function _getHelperFunctions() {
		$utilReflection = new \ReflectionClass('TemplateUtil');
		$utilMethods = $utilReflection->getMethods(ReflectionMethod::IS_PUBLIC);

		$methods = array('N/A' => 'N/A');
		foreach ($utilMethods as $item) {
			if ($item->name === '__construct') {
				continue;
			}
			// If the method name is prefixed admin we want to show it as the non-admin form in the list.
			$methodName = lcfirst(str_replace('admin', '', $item->name));
			$methods[$methodName] = $methodName;
		}

		return $methods;
	}

	/**
	 * return array of block fields available (alt, caption etc..)
	 *
	 * @return array 	array of available fields
	 */
	protected function _getBlockFields() {
		return array(
			'ImageBlock' => array(
				array(
					'value' => 'alt',
					'name' => 'Alt',
					'class' => 'protect-value' // sets class on the input checkbox to prevent js cloneable from removing value
				),
				array(
					'value' => 'caption',
					'name' => 'Caption',
					'class' => 'protect-value' // sets class on the input checkbox to prevent js cloneable from removing value
				),
				array(
					'value' => 'link_text',
					'name' => 'Link Text',
					'class' => 'protect-value' // sets class on the input checkbox to prevent js cloneable from removing value
				),
				array(
					'value' => 'link_url',
					'name' => 'Link Url',
					'class' => 'protect-value' // sets class on the input checkbox to prevent js cloneable from removing value
				)
			),
			'DocumentBlock' => array(
				array(
					'value' => 'caption',
					'name' => 'Caption',
					'class' => 'protect-value' // sets class on the input checkbox to prevent js cloneable from removing value
				)
			)
		);
	}

	/**
	 * find all the templates within the view folder
	 *
	 * @return array 	array of templates to populate select
	 */
	protected function _getSelectableTemplates() {
		// search for templates
		$templates = array();

		$templateDir = 'Templates';
		$viewDir = APP . 'View' . DS;
		$templates = $this->_processSelectableTemplates($viewDir, $templateDir);

		// find all plugin templates
		$plugins = App::objects('plugins');

		foreach ($plugins as $plugin) {
			if ($plugin === 'EvTemplates') {
				continue;
			}

			$pluginViewDir = CakePlugin::path($plugin) . 'View' . DS;

			// extended plugin dir
			$extendedPluginViewDir = $viewDir . DS . 'Plugin' . DS . $plugin . DS;

			// merge together template files found for app based templates, plugin
			// based templates and extended plugin based templates
			$templates = Hash::merge(
				$templates,
				$this->_processSelectableTemplates(
					$pluginViewDir,
					$templateDir,
					$plugin
				),
				$this->_processSelectableTemplates(
					$extendedPluginViewDir,
					$templateDir,
					$plugin
				)
			);
		}

		return $templates;
	}

	/**
	 * given the paths, perform the actual template search
	 * format and return array
	 *
	 * @param 	string 			Absolute path to the View dir to check
	 * @param 	string 			Templates directory (should always be 'Templates')
	 * @param 	string|bool 	Plugin to search in
	 * @return  array
	 */
	protected function _processSelectableTemplates($path, $templateDir, $plugin = false) {
		$dir = new Folder($path . $templateDir);
		$files = $dir->findRecursive('.*\.ctp');

		$templates = array();

		foreach ($files as $file) {

			$filePath = str_replace($path . $templateDir . DS, '', $file);
			$filename = substr($filePath, 0, -4);

			if ($plugin !== false) {
				$templates[$plugin][$plugin . './' . $templateDir . DS . $filename] = $filePath;
			} else {
				$templates[DS . $templateDir . DS . $filename] = $filePath;
			}
		}

		return $templates;
	}

	/**
	 * Prepare the admin form fields for the menu item form
	 * @return array
	 */
	protected function _adminFormFields() {
		$Model = $this->{$this->modelClass};

		$fields = parent::_adminFormFields();

		$fields[$Model->alias . '.template']['type'] = 'select';
		$fields[$Model->alias . '.helperFunction']['type'] = 'select';
		$fields[$Model->alias . '.cfDataLabel']['label'] = 'Custom Data Label';
		$fields[$Model->alias . '.model']['label'] = 'Restrict to model';

		return $fields;
	}

	/**
	 * redefine admin_edit to set our own template
	 *
	 */
	public function admin_edit($id = null) {
		if (
			! empty($this->request->data) &&
			(
				$this->request->is('post') ||
				$this->request->is('put')
			)
		) {
			// check for image / document blocks to delete
			if (! empty($this->request->data['ImageBlock']['toDelete'])) {
				$this->_deleteBlocks('Image');
			}

			if (! empty($this->request->data['DocumentBlock']['toDelete'])) {
				$this->_deleteBlocks('Document');
			}

			unset(
				$this->request->data['ImageBlock']['toDelete'],
				$this->request->data['DocumentBlock']['toDelete']
			);
		}

		parent::admin_edit($id);

		$this->view = 'EvTemplates.Templates/admin_form';
	}

	/**
	 * delete any images / document blocks that have been marked for delete
	 *
	 * @param 	string 		Type of block to delete
	 */
	protected function _deleteBlocks($type) {
		$ids = explode(',', $this->request->data[$type . 'Block']['toDelete']);
		$ids = ArrayUtil::clearArray($ids);

		$Model = EvClassRegistry::init('EvTemplates.' . $type . 'Block');
		$Model->deleteAll(
			array(
				$type . 'Block.id' => $ids
			),
			true,
			true
		);
	}

	/**
	 * called via ajax
	 * given the template ID get all the fields and process in template
	 * so we can fill out the data
	 *
	 * @param 	int 	Template id
	 * @return 	string 	processed html containing the fields
	 */
	public function ajax_getFieldsForTemplate($templateId = null, $modelName = null, $modelId = null) {
		$this->view = 'EvTemplates.Templates/fields';
		$this->layout = 'ajax';

		if (empty($templateId)) {
			return;
		}

		$template = $this->Template->readForView($templateId);

		// Check if an admin helper function has been defined for this template.
		if (! empty($template['Template']['helperFunction'])) {
			$TemplateUtil = new TemplateUtil($this);
			$helper = 'admin' . ucfirst($template['Template']['helperFunction']);
			if (method_exists($TemplateUtil, $helper)) {
				$template = $TemplateUtil->$helper($template, $this->request->data);
			}
		}

		// search for the model data so we can prepop any fields we find
		if (! empty($modelName) && ! empty($modelId)) {
			$Model = EvClassRegistry::init($modelName);
			$this->request->data = array_merge_recursive(
				$this->request->data,
				$Model->find(
					'first',
					array(
						'conditions' => array(
							$Model->alias . '.id' => $modelId
						),
						'customFields' => true
					)
				)
			);
		}

		if (!empty($template['CustomFields'])) {
			// Assign displayInfo field if there is display_info data present.
			// Required as saving custom fields instead of saving NULL it will make an empty string
			foreach ($template['CustomFields'] as $fieldId => &$field) {
				if (!empty($field['Field']['display_info'])) {
					$field['Field']['displayInfo'] = $field['Field']['display_info'];
				}
			}
		}

		$this->set('template', $template);

		if ($this->request->is('post') && ! empty($this->request->data['validationErrors'])) {
			$CustomFieldData = EvClassRegistry::init('EvCustomFields.CustomFieldData');
			$CustomFieldData->validationErrors = $this->request->data['validationErrors'];
		}
	}

	/**
	 * called via ajax
	 * given the template ID get all the fields and process in template
	 * so we can fill out the data
	 *
	 * @param 	int 	Template id
	 * @return 	string 	processed html containing the fields
	 */
	public function ajax_getMediaForTemplate($templateId = null, $modelName = null, $modelId = null) {
		$this->view = 'EvTemplates.Templates/img_docs';
		$this->layout = 'ajax';
		$this->theme = 'admin';

		if (empty($templateId) || empty($modelName)) {
			return;
		}

		$Model = EvClassRegistry::init($modelName);
		$modelInfo = pluginSplit(EvClassRegistry::getNameFromModel($Model));

		// calling this behavior method through Model will auto add Model var as first param.
		$Model->setupAttachmentRelationships($templateId);

		//contain all the image / document slots to load oin template
		$query = $Model->containSlotAssociations(
			array(
				'Image' => $Model->imageSlots,
				'Document' => $Model->documentSlots
			)
		);

		// add extra template_id condition to make sure we're brining out the correct media
		foreach ($Model->hasMany as $key => $relationship) {
			if (in_array($relationship['className'], array('EvCore.Image', 'EvCore.Document', 'EvCoreImage', 'EvCoreDocument'))) {
				$Model->hasMany[$key]['conditions'][$key . '.template_id'] = $templateId;
			}
		}

		// search for the model data so we can prepop any fields we find
		if (! empty($modelName) && ! empty($modelId)) {
			$query['conditions'] = array(
				$Model->alias . '.id' => $modelId
			);

			$this->request->data = $Model->find(
				'first',
				$query
			);
		}

		$this->set(
			array(
				'imageSlots' => $Model->imageSlots,
				'documentSlots' => $Model->documentSlots,
				'actualModel' => get_class($Model),
				'plugins' => App::objects('plugin'),
				'model' => $Model->alias,
				'primaryModelPlugin' => (! empty($modelInfo['0'])) ? $modelInfo['0'] : null,
				'primaryModel' => (! empty($modelInfo['1'])) ? $modelInfo['1'] : null

			)
		);
	}

	/**
	 * called via ajax
	 * given the template ID get the tab name for the custom fields
	 *
	 * @param 	int 	Template Id
	 * @return 	string
	 */
	public function ajax_getTabNameFromTemplate($templateId = null) {
		if (! empty($templateId)) {
			$label = $this->Template->field('cfDataLabel', array('Template.id' => $templateId));
		} else {
			$label = 'Custom Data';
		}

		$this->response->type('json');
		$this->response->disableCache();
		$this->response->body(
			json_encode(
				array('label' => $label)
			)
		);
		return $this->response;
	}

	/**
	 * Admin template preview image
	 *
	 * Used to show a preview image of what a template looks like.
	 * @param  integer $templateId Id of the template to load the image for
	 * @return void
	 */
	public function admin_tpl_preview_img($templateId) {
		// Check to see if there's an image available
		$this->loadModel('EvCore.Image');

		$image = $this->Image->find('first', array(
			'conditions' => array(
				'model' => 'Template',
				'model_id' => $templateId,
				'attachment_type' => 'PreviewImage'
			)
		));

		$this->set('image', $image);
		$this->layout = false;
		$this->view = 'EvTemplates.tpl_preview_image';
	}
}
