<?php

App::uses('AppBehavior', 'Model/Behavior');

class CalculateTaxBehavior extends AppBehavior {

/**
 * Link the model that has this behavior with the tax levels model so we can contain data.
 *
 * @param Model $Model The model that called this behavior.
 * @return void.
 */
	public function linkTaxCalc(Model $Model) {
		$Model->bindModel(
			array(
				'belongsTo' => array(
					'TaxLevel' => array(
						'className' => 'EvTax.TaxLevel'
					)
				)
			),
			false
		);
	}

/**
 * Initialise the behavior - store the settings against the model
 *
 * @param Model $Model    The model that called this behavior.
 * @param array $settings The behavior settings to use on this model.
 * @return void.
 * @see ModelBehavior::setup()
 */
	public function setup(Model $Model, $settings = array()) {
		$defaults = array(
			'fields' => array(),
			'TaxRateModel' => false,
			'TaxRateModelId' => false
		);

		$this->settings[$Model->alias] = array_merge(
			$defaults,
			$settings
		);

		if (empty($this->settings[$Model->alias]['TaxRateModel'])) {
			$this->linkTaxCalc($Model);
		}
	}

/**
 * beforeFind can be used to cancel find operations, or modify the query that will be executed.
 * By returning null/false you can abort a find. By returning an array you can modify/replace the query
 * that is going to be run.
 *
 * @param Model $Model Model using this behavior
 * @param array $query Data used to execute this query, i.e. conditions, order, etc.
 * @return bool|array False or null will abort the operation. You can return an array to replace the
 *   $query that will be eventually run.
 */
	public function beforeFind(Model $Model, $query) {
		if (empty($this->settings[$Model->alias]['TaxRateModel'])) {
			if (! isset($query['contain']) || (! isset($query['contain']['TaxLevel']) && array_search('TaxLevel', $query['contain']) === false)) {
				$query['contain'][] = 'TaxLevel';
			}
		}

		return $query;
	}

/**
 * After find callback. Can be used to modify any results returned by find.
 *
 * @param Model $Model Model using this behavior
 * @param mixed $results The results of the find operation
 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
 * @return mixed An array value will replace the value of $results - any other value will be ignored.
 */
	public function afterFind(Model $Model, $results, $primary = false) {
		if (empty($this->settings[$Model->alias]['fields']) || ! is_array($this->settings[$Model->alias]['fields'])) {
			return $results;
		}

		$TaxRateModel = null;
		if (! empty($this->settings[$Model->alias]['TaxRateModel'])) {
			$TaxRateModel = EvClassRegistry::init($this->settings[$Model->alias]['TaxRateModel']);
		}

		foreach ($results as $key => $item) {
			$results[$key] = $this->calculateTax($Model, $item, $TaxRateModel);
		}

		return $results;
	}

/**
 * Calculate the taxed amounts for the item.
 *
 * @param Model $Model        Model using this behavior.
 * @param array $item         The item we wish to calculate tax for.
 * @param obj   $TaxRateModel The model from which we can get the tax rate.
 * @return array $item The item with additiona _incTax fields.
 */
	public function calculateTax(Model $Model, $item, $TaxRateModel = null) {
		/*
		 * The item can be passed through from the result of a find() or from somewhere else. This means we need to
		 * extract the fields to be calculated on and the tax rate.
		 */
		$fields = [];
		$multipleRecords = false;
		if (!empty($item[$Model->alias]) && is_array($item[$Model->alias]) && isset(current($item[$Model->alias])['id'])) {
			//Calculate the tax on multiple children of the item (Variant pricing of variants).
			$fields = $item[$Model->alias];
			$multipleRecords = true;
		} elseif (isset($item[$Model->alias])) {
			//Calculate the tax on a single model item.
			$fields = $item[$Model->alias];
		} else {
			//Calculate the tax on a single item.
			$fields = $item;
		}

		if (empty($fields) || !is_array($fields)) {
			/*
			 * If no fields could be found or a single found has been provided then return the item as tax can't be
			 * calculated. Use addTaxToPrice() if a single field wants tax added to it.
			 */
			return $item;
		}

		if ($multipleRecords) {
			foreach ($fields as &$subFields) {
				$subFields = $this->_addTaxToFields($Model, $subFields, $TaxRateModel);
			}

			$item[$Model->alias] = $fields;
		} else {
			$fields = $this->_addTaxToFields($Model, $fields, $TaxRateModel);

			if (isset($item[$Model->alias])) {
				$item[$Model->alias] = $fields;
			} else {
				$item = $fields;
			}
		}

		return $item;
	}

/**
 * Calculate the tax value for each of the fields in the behavior settings.
 *
 * @param Model $Model        The model current having tax added to it's fields.
 * @param array $fields       The fields to have tax added to.
 * @param Model $TaxRateModel The model to get the tax rate from.
 * @return array The fields with incTax fields added.
 */
	protected function _addTaxToFields($Model, $fields, $TaxRateModel = null) {
		// Work out tax level
		$rate = 0.00;
		if (!empty($fields['TaxLevel']['rate'])) {
			$rate = $fields['TaxLevel']['rate'];
		} elseif (!empty($TaxRateModel)) {
			$rate = $this->_getRate($Model, $TaxRateModel, $fields);
		} elseif (method_exists($Model, 'getTaxRate') || $Model->Behaviors->enabled('Taxable')) {
			//If a tax rate model hasn't been provided then attempt to get the tax rate from the current model.
			$rate = $this->_getRate($Model, $Model, $fields);
		}

		foreach ($this->settings[$Model->alias]['fields'] as $field) {
			if (!isset($fields[$field])) {
				continue;
			}

			$fields[$field . '_incTax'] = $this->addTaxToPrice($this, $fields[$field], $rate);
		}

		return $fields;
	}

/**
 * Add a taxed amount to a value based on a tax rate.
 *
 * @param Model     $Model The model tax is being added to.
 * @param int|float $price The value to add tax to.
 * @param float     $rate  The tax rate as a percentage.
 * @return int|float The taxed valued.
 */
	public function addTaxToPrice($Model, $price, $rate = 0.00) {
		return $price + ($price * ($rate / 100));
	}

/**
 * work out the tax rate
 * - either query table
 * - retrieve from stored array
 *
 * @param Model $Model        The original model that called this behavior.
 * @param Model $TaxRateModel The related model that has the tax rate id.
 * @param array $item         The record to get a tax rate for.
 * @return decimal Decimal rate, will return zero if none found
 */
	protected function _getRate($Model, $TaxRateModel, $item) {
		$modelIdField = $this->settings[$Model->alias]['TaxRateModelId'];
		if (empty($modelIdField)) {
			$modelIdField = $Model->primaryKey;
		}

		$modelId = Hash::get($item, $modelIdField);

		if (! empty($this->settings[$Model->alias]['taxRateCache'][$modelId])) {
			return $this->settings[$Model->alias]['taxRateCache'][$modelId];
		}

		if (! empty($modelId)) {
			$rate = $TaxRateModel->getTaxRate($modelId);

			if (! empty($rate)) {
				$this->settings[$Model->alias]['taxRateCache'][$modelId] = $rate;
				return $rate;
			}
		}

		return intval(0);
	}
}
