<?php

use Omnipay\Omnipay;

App::uses('PaypalLib', 'EvPaypal.Lib');
App::uses('GatewayBaseComponent', 'EvTransactions.Controller/Component');
App::uses('GatewayInterface', 'EvTransactions.Lib');
App::uses('CakeText', 'Utility');

class StripeComponent extends GatewayBaseComponent implements GatewayInterface {

	protected $_controller = null;

	protected $_api = null;

	public $Transaction = null;

	protected $_params = [];

	public function initialize(Controller $controller) {
		parent::initialize($controller);
		$this->_controller = $controller;
		$this->_api = Omnipay::create('Stripe\PaymentIntents');
		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		// Load the Stripe config here rather than the `setup()` method as we need these settings in
		// `beforeRender()` which is called before setup.
		if (Configure::read('app.environment') === 'PRODUCTION') {
			$this->_config = Configure::read('EvTransactions.stripe.live');
		} else {
			$this->_config = Configure::read('EvTransactions.stripe.dev');
		}
	}

	/**
	 * init
	 *
	 * Use the function setup and connection / settings that need setting up in the gateway
	 */
	public function setup() {
		// Set the API key.
		$this->_api->setApiKey($this->_config['secret_key']);
	}

	/**
	 * Before render. Sets the publishable key to a view variable.
	 * @param  Controller $controller
	 * @return void
	 */
	public function beforeRender(Controller $controller) {
		$controller->set('stripePublishableKey', $this->_config['publishable_key']);
	}

/**
 * setupPayment
 *
 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
 *
 * @param int - transaction id we have created
 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
 * @param array $items - multidimensional array break down of the transaction items, required elements are 'description' and 'amount'
 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = array()) {
		$this->_params = [];
		$this->Transaction->id = $transactionId;

		$this->_params = [
			'amount' => is_array($amount) ? $amount['amount'] : $amount,
			'confirm' => true,
			'confirmation_method' => 'automatic',
			'currency' => is_array($amount) ? $amount['currency'] : Configure::read('EvTransactions.currency'),
			'returnUrl' => Router::url($return['return'], true),
			'source' => $this->_controller->request->data['stripeToken'] ?? null,
			'paymentMethod' => $this->_controller->request->data['PaymentMethodId'] ?? null,
		];
	}

	/**
	 * getPayment
	 *
	 * Everything should be setup, actually take the payment
	 *
	 * @param int transactions id
	 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
	 */
	public function getPayment($transactionId) {
		try {
			$response = $this->_api->purchase($this->_params)->send();
			$paymentIntentReference = $response->getPaymentIntentReference();

			if ($response->isSuccessful()) {
				// The payment has been successfully taken, so store the payment ID.
				$data = [
					'id' => $transactionId,
					'status' => 'success',
					'message' => $response->getMessage(),
					'payment_token' => $response->getData()['id'],
					'payment_intent_ref' => $paymentIntentReference,
				];
				$this->Transaction->save($data);

			} elseif ($response->isRedirect()) {
				$data = [
					'id' => $transactionId,
					'status' => 'initiated',
					'message' => $response->getMessage(),
					'payment_token' => $response->getData()['id'],
					'payment_intent_ref' => $paymentIntentReference,
				];
				$this->Transaction->save($data);
				return $response->redirect();
			} else {
				// Payment has failed.
				$data = [
					'id' => $transactionId,
					'status' => 'failed',
					'message' => $response->getMessage(),
					'payment_intent_ref' => $paymentIntentReference,
				];
			}
			$this->Transaction->save($data);
		} catch (\Exception $e) {
			throw new InternalErrorException(
				__d('ev_stripe', 'Sorry, there was an error processing your payment. Please try again later.')
			);
		}

		// Unlike most payment gateways that would now be sending you off to their site for payment
		// and redirecting back to a callback we have already processed the payment. Therefore, we
		// return the status of the payment here rather than via `processReturn()`.
		return [
			'result' => $response->isSuccessful(),
			'message' => $response->getMessage(),
			'transaction_id' => $transactionId,
			'payment_intent_ref' => $paymentIntentReference,
		];
	}

/**
 * Authorise payment returned from stripe
 * When the user has finished their 2FA checks, they will be returned to this route on the website
 *
 * Check there's a transaction with the same payment_intent reference in the database,
 * if not, return false
 * if there is, confirm the payment and complete the transaction
 *
 * @param array $return a list of return data, expects a URL
 * @return array
 */
	public function authorisePayment(array $return) {
		if (empty($this->_controller->request->query['payment_intent']) && !empty($this->_controller->request->query['transaction'])) {
			if (empty($this->_controller->request->query['transaction'])) {
				return [
					'result' => false
				];
			}
			// Doesn't need authorisation
			$this->_controller->redirect($return['return'] . '?transaction=' . $this->_controller->request->query['transaction']);
		}
		$paymentIntent = $this->_controller->request->query['payment_intent'];

		// Check there's a transaction in the database with this paymentIntent reference
		$transaction = $this->Transaction->findByPaymentIntentRef($paymentIntent);
		$transactionId = $transaction['Transaction']['id'];

		if (empty($transaction['Transaction'])) {
			return [
				'result' => false
			];
		}

		// When we return to the payment complete page, it needs the transaction ID to check against
		$returnUrl = $return['return'] . '?transaction=' . $transactionId;

		try {
			$response = $this->_api->confirm([
				'paymentIntentReference' => $paymentIntent,
				'returnUrl' => $returnUrl,
			])->send();

			if ($response->isSuccessful()) {
				$this->Transaction->save([
					'id' => $transactionId,
					'status' => 'success',
				]);
			} else {
				$this->Transaction->save([
					'id' => $transactionId,
					'status' => 'failed',
				]);
				return [
					'result' => false,
					'returnUrl' => $returnUrl,
				];
			}
		} catch (\Exception $e) {
			throw new InternalErrorException(
				__d('ev_stripe', 'Sorry, there was an error processing your payment. Please try again later.')
			);
		}

		return [
			'result' => $response->isSuccessful(),
			'returnUrl' => $returnUrl,
			'transaction_id' => $transactionId,
			'payment_intent_ref' => $paymentIntent,
		];
	}

/**
 * Stripe has already taken the money (or not) by this point and doesn't have a callback currently
 * so the return is used when checkPayment is called to check the transaction was successful
 *
 * @return array
 */
	public function processReturn() {
		// Fetch the transaction using the ID
		// If we haven't been passed on we can't return a success
		if (empty($this->_controller->request->query['transaction'])) {
			return [
				'result' => false
			];
		}
		$transactionId = $this->_controller->request->query['transaction'];

		// All the processing has already been done by this point so we just need to check if the
		// transaction was successful
		$transaction = $this->Transaction->findById($transactionId);

		// No transaction it can't have been a success
		if (empty($transaction)) {
			return [
				'result' => false
			];
		}

		// Return the result of the transaction
		return [
			'result' => $transaction['Transaction']['status'] == 'success' ? true : false
		];
	}

}
