<?php

use Omnipay\Omnipay;

App::uses('GatewayBaseComponent', 'EvTransactions.Controller/Component');
App::uses('GatewayInterface', 'EvTransactions.Lib');

class StripeCheckoutComponent extends GatewayBaseComponent implements GatewayInterface {

	protected $_controller = null;

	protected $_api = null;

	public $Transaction = null;

	protected $_params = [];

	public function initialize(Controller $controller) {
		parent::initialize($controller);
		$this->_controller = $controller;
		$this->_api = Omnipay::create('Stripe\Checkout');
		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		// Load the Stripe config here rather than the `setup()` method as we need these settings in
		// `beforeRender()` which is called before setup.
		if (Configure::read('app.environment') === 'PRODUCTION') {
			$this->_config = Configure::read('EvTransactions.stripe.live');
		} else {
			$this->_config = Configure::read('EvTransactions.stripe.dev');
		}
	}

/**
 * init
 *
 * Use the function setup and connection / settings that need setting up in the gateway
 */
	public function setup() {
		// Set the API key.
		$this->_api->setApiKey($this->_config['secret_key']);
	}

/**
 * Before render. Sets the publishable key to a view variable.
 * @param  Controller $controller
 * @return void
 */
	public function beforeRender(Controller $controller) {
		$controller->set('stripePublishableKey', $this->_config['publishable_key']);
	}

/**
 * setupPayment
 *
 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
 *
 * @param int - transaction id we have created
 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
 * @param array $items - multidimensional array break down of the transaction items, required elements are 'description' and 'amount'
 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = []) {
		$this->_params = [];
		$this->Transaction->id = $transactionId;

		$currency = is_array($amount) ? $amount['currency'] : Configure::read('EvTransactions.currency');

		$lineItems = [];

		// format the items into stripe price data. The unit_amount has to be converted to pennies for stripe
		foreach ($items as $item) {
			$lineItems[] = [
				'price_data' => [
					'currency' => $currency,
					'unit_amount' => number_format($item['amount'], 2) * 100,
					'product_data' => [
						'name' => $item['description'],
					],
				],
				'quantity' => 1,
			];
		}

		$this->_params = [
			'amount' => is_array($amount) ? $amount['amount'] : $amount,
			'confirm' => true,
			'confirmation_method' => 'automatic',
			'currency' => $currency,
			'success_url' => Router::url($return['return'], true),
			'cancel_url' => Router::url($return['cancel'], true),
			'source' => $this->_controller->request->data['stripeToken'] ?? null,
			'paymentMethod' => $this->_controller->request->data['PaymentMethodId'] ?? null,
			'line_items' => $lineItems,
			'mode' => 'payment',
		];

		$this->_params = array_merge($this->_params, $extra);
	}

/**
 * getPayment
 * Everything should be setup, actually take the payment. We set up the
 * Stripe Checkout session and redirect the user to the payment page. Stripe
 * handles everything on their end.
 *
 * @param int transactions id
 * @return mixed redirect to the stripe checkout page, or call the processReturn function to handle a response
 */
	public function getPayment($transactionId) {
		try {
			$transaction = $this->Transaction->findById($transactionId);

			if ($transaction['Transaction']['status'] === 'initiated' && $transaction['Transaction']['payment_token'] == '') {
				$response = $this->_api->purchase($this->_params)->send();
				$data = $response->getData();

				// Set the transaction details - these are provided by the gateway
				// regardless of the status, so we can use them to update the
				// transaction record
				$paymentIntentReference = $data['payment_intent'] ?? null;

				$transactionData = [
					'id' => $transactionId,
					'status' => 'initiated',
					'message' => $response->getMessage(),
					'payment_token' => $data['id'] ?? null,
					'payment_intent_ref' => $paymentIntentReference,
				];
				$this->Transaction->save($transactionData);

				if (! empty($data['url'])) {
					return $this->_controller->redirect($data['url']);
				}
			}

			return $this->processReturn();
		} catch (\Exception $e) {
			throw new InternalErrorException(
				__d('ev_stripe', 'Sorry, there was an error processing your payment. Please try again later.')
			);
		}
	}

/**
 * Once redirected back from the Stripe Checkout this method should be called to
 * finalise the transaction on the website. It will check the status of the
 * transaction on the Stripe payment and update the transaction record.
 *
 * @return array Return an array of response data, the key being the 'result' value, which will be true or false
 */
	public function processReturn() {
		// Fetch the transaction using the ID
		// If we haven't been passed on we can't return a success
		if (empty($this->_controller->request->query['transaction'])) {
			return [
				'result' => false
			];
		}
		$transactionId = $this->_controller->request->query['transaction'];

		// All the processing has already been done by this point so we just need to check if the
		// transaction was successful
		$transaction = $this->Transaction->findById($transactionId);

		// No transaction it can't have been a success
		if (empty($transaction)) {
			return [
				'result' => false
			];
		}

		// Get the payment from stripe using the payment token (aka checkout session ref)
		$paymentToken = $transaction['Transaction']['payment_token'];
		$transaction = $this->_api->fetchTransaction();
		$transaction->setTransactionReference($paymentToken);
		$response = $transaction->send();
		$data = $response->getData();

		$paymentIntentReference = $data['payment_intent'] ?? null;

		// If the payment is successful, we can return a success
		if ($data['status'] == 'complete') {
			$data = [
				'id' => $transactionId,
				'status' => 'success',
				'message' => $response->getMessage() ?? null,
				'payment_token' => $paymentToken,
				'payment_intent_ref' => $paymentIntentReference,
			];
			$this->Transaction->save($data);

			return [
				'result' => true,
				'transaction_id' => $transactionId,
				'payment_intent_ref' => $data['payment_intent'] ?? null,
				'payment_token' => $paymentToken,
			];
		}

		$this->Transaction->save([
			'id' => $transactionId,
			'status' => 'failed',
		]);

		// If the payment is not successful, we can return a failure
		return [
			'result' => false,
			'transaction_id' => $transactionId,
			'payment_intent_ref' => $data['payment_intent'] ?? null,
			'payment_token' => $paymentToken,
		];
	}
}
