<?php

declare(strict_types=1);

namespace Tests\Unit;

use Carbon\CarbonImmutable;
use Evoluted\StatamicTwig\Twig\StatamicExtension;
use Evoluted\StatamicTwig\Twig\StatamicRuntimeLoader;
use PHPUnit\Framework\TestCase;
use Twig\Environment;
use Twig\Loader\ArrayLoader;

class FiltersTest extends TestCase
{
    /**
     * @dataProvider filtersProvider
     */
    public function testFilters(string $twigTemplate, string $expectedOutput): void
    {
        $loader = new ArrayLoader([ 'test.html.twig' => $twigTemplate ]);
        $twig = new Environment($loader, [
            'debug' => false,
            'cache' => false,
            'strict_variables' => true,
            'autoescape' => 'name',
            'optimizations' => 0,
        ]);
        $twig->addExtension(new StatamicExtension());
        $twig->addRuntimeLoader(new StatamicRuntimeLoader());

        // some filters (e.g. obfuscate_email) use rand() so make it deterministic
        srand(1234567890);

        $this->assertSame($expectedOutput, $twig->render('test.html.twig'));
    }

    public function filtersProvider(): array
    {
        $today = CarbonImmutable::now();
        $fiveYearsAgo = new CarbonImmutable($today->subYears(5));
        $fiveMonthsAgo = new CarbonImmutable($today->subMonths(5));
        $fiveWeeksAgo = new CarbonImmutable($today->subWeeks(5));
        $fiveDaysAgo = new CarbonImmutable($today->subDays(5));
        $fiveHoursAgo = new CarbonImmutable($today->subHours(5));
        $fiveMinutesAgo = new CarbonImmutable($today->subMinutes(5));
        $fiveSecondsAgo = new CarbonImmutable($today->subSeconds(5));
        $yesterday = new CarbonImmutable($today->subDay());
        $tomorrow = new CarbonImmutable($today->addDay());
        return [
            'add_slashes' => [ "{{ '\"I\\'m not listening!\" said the small, strange creature.' | add_slashes }}", '\\"I\\\'m not listening!\\" said the small, strange creature.' ],
            'ampersand_list' => [ "{{ [ 'apples', 'bananas', 'jerky' ] | ampersand_list }}", 'apples, bananas & jerky' ],
            'ampersand_list w/params' => [ "{{ [ 'apples', 'bananas', 'jerky' ] | ampersand_list('and') }}", 'apples, bananas and jerky' ],
            'ascii' => [ "{{ 'lemoñade' | ascii }}", 'lemonade' ],
            'at' => [ "{{ 'supercalifragilisticexpialidocious' | at(21) }}", "x" ],
            'background_position' => [ "{{ '50-30' | background_position }}", '50% 30%' ],
            'backspace' => [ "{{ 'supercalifragilisticexpialidocious'| backspace(29) }}", 'super' ],
            'camelize' => [ "{{ 'make_everything_better' | camelize }}", 'makeEverythingBetter' ],
            'cdata' => [ "{{ 'My Very Own Podcast' | cdata }}", '<![CDATA[My Very Own Podcast]]>' ],
            'collapse' => [ "{{ [ ['one', 'two', 'three'], ['four', 'five', 'potato'] ] | collapse | ampersand_list }}", 'one, two, three, four, five & potato' ],
            'collapse_whitespace' => [ "{{ 'Bad   at           typing  ' | collapse_whitespace }}", 'Bad at typing' ],
            'contains_all' => [ "{{ 'It was the best of times, it was the worst of times.' | contains_all('best', 'worst') }}", '1' ],
            'contains_all 2' => [ "{{ 'It was the best of times, it was the worst of times.' | contains_all('best', 'better') }}", '' ],
            'contains_any' => [ "{{ 'It was the best of times, it was the worst of times.' | contains_any('good', 'better', 'best') }}", '1' ],
            'count_substring' => [ "{{ 'Dude! You got a tattoo! So do you, dude! Dude, what does my tattoo say? Sweet! What about mine? Dude! What does mine say? Sweet! What about mine? Dude! What does mine say?' | count_substring(\"dude\") }}", '5' ],
            'dashify' => [ "{{ 'Just Because I Can' | dashify }}", 'just-because-i-can' ],
            'days_ago' => [ "{{ '" . $fiveDaysAgo->format('F d Y H:i:s') . "' | days_ago }}", '5'],
            'deslugify' => [ "{{ 'Just-Because-I-Can' | deslugify }}", 'Just Because I Can' ],
            'embed_url' => [ "{{ 'https://www.youtube.com/watch?v=s9F5fhJQo34' | embed_url }}", 'https://www.youtube-nocookie.com/embed/s9F5fhJQo34' ],
            'ends_with' => [ "{{ 'That\'s what she said!' | ends_with('she said!') }}", '1' ],
            'ensure_left' => [ "{{ 'statamic.com' | ensure_left('http://') }}", 'http://statamic.com' ],
            'ensure_right' => [ "{{ 'statamic' | ensure_right('.com') }}", 'statamic.com' ],
            'excerpt' => [ "{{ 'Lorem Ipsum dolor sit amet.<!--more--> consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Massa id neque aliquam vestibulum.' | excerpt }}", 'Lorem Ipsum dolor sit amet.'],
            'excerpt w/params' => [ "{{ 'Lorem Ipsum dolor sit amet.<!--more--> consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Massa id neque aliquam vestibulum.<!-- end -->' | excerpt('<!-- end -->') }}", 'Lorem Ipsum dolor sit amet.&lt;!--more--&gt; consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Massa id neque aliquam vestibulum.'],
            'explode' => [ "{{ 'Scotland, England, Switzerland, Italy' | explode(', ') | ampersand_list }}", 'Scotland, England, Switzerland & Italy'],
            'favicon' => [ "{{ '/assets/img/favicon.png' | favicon }}", '<link rel="shortcut icon" type="image/x-icon" href="/assets/img/favicon.png">' ],
            'flatten' => [ "{{ { 'spices': [ 'garlic', 'cumin', 'ginger', 'turmeric', 'paprika', 'curry powder' ], 'vegetables': [ 'tomatoes', 'onion' ], 'meat': [ 'chicken' ] } | flatten | join(',')  }}", 'garlic,cumin,ginger,turmeric,paprika,curry powder,tomatoes,onion,chicken' ],
            'flip' => [ "{{ { 'food': 'burger', 'drink': 'soda' } | flip | json_encode | raw }}", '{"burger":"food","soda":"drink"}' ],
            'format' => [ "{{ 'April 15 2016' | format(\"Y-m-d\") }}", '2016-04-15' ],
            'format_number' => ["{{ 130134.109 | format_number }}", '130,134'],
            'format_number w/params' => ["{{ 130134.109 | format_number(1, ',', ',') }}", '130,134,1'],
            'has_lower_case' => [ "{{ 'I DON\'T KNOW WHAT WE\'RE YELLING ABOUT!' | has_lower_case }}", '' ],
            'has_upper_case' => [ "{{ 'I DON\'T KNOW WHAT WE\'RE YELLING ABOUT!' | has_upper_case }}", '1' ],
            'hours_ago' => ["{{ '" . $fiveHoursAgo->format('F d Y h:i:s') . "' | hours_ago }}", '5'],
            'image' => ["{{ '/assets/img/bokeh-bunnies.jpg' | image }}", '<img src="/assets/img/bokeh-bunnies.jpg">'],
            'insert' => ["{{ 'This is yummy.' | insert('not ', 8) }}", 'This is not yummy.'],
            'is_after' => [ "{{ '2021-01-01 00:00:01' | is_after('2021-01-01 00:00:00') }}", '1' ],
            'is_after fail' => [ "{{ '2021-01-01 00:00:00' | is_after('2021-01-01 00:00:01') }}", '' ],
            'is_alpha' => ["{{ 'abcdefg' | is_alpha }}", '1'],
            'is_alpha fail' => ["{{ 'abc123' | is_alpha }}", ''],
            'is_alphanumeric' => ["{{ 'abc123' | is_alphanumeric }}", '1'],
            'is_alphanumeric fail' => ["{{ 'abc123!@#' | is_alphanumeric }}", ''],
            'is_array' => ["{{ [ 'apples', 'bananas', 'jerky' ] | is_array }}", '1'], // No example
            'is_before' => [ "{{ '2021-01-01 00:00:00' | is_before('2021-01-01 00:00:01') }}", '1' ],
            'is_before fail' => [ "{{ '2021-01-01 00:00:01' | is_before('2021-01-01 00:00:00') }}", '' ],
            'is_between' => [ "{{ '2021-01-01 00:00:00' | is_between('2020-12-31 23:59:59', '2021-01-01 00:00:01') }}", '1' ],
            'is_between fail' => [ "{{ '2021-01-01 00:00:00' | is_between('2021-01-01 00:00:01', '2021-01-01 00:00:02') }}", '' ],
            'is_blank' => ["{{ '           ' | is_blank }}", '1'],
            'is_blank fail' => ["{{ '     potato      ' | is_blank }}", ''],
            'is_email' => ["{{ 'lknope@inpra.org' | is_email }}", '1'],
            'is_email fail' => ["{{ 'waffles' | is_email }}", ''],
            'is_embeddable' => ["{{ 'https://www.youtube.com/watch?v=s9F5fhJQo34' | is_embeddable }}", '1'],
            'is_embeddable fail' => ["{{ 'http://example.com/video.mp4' | is_embeddable }}", ''],
            'is_empty' => ["{{ ['', '', ''] | is_empty }}", '1'], // No example
            'is_future' => ["{{ '" . $tomorrow->format('F d Y H:i:s') . "' | is_future }}", '1'],
            'is_future fail' => ["{{ '" . $fiveDaysAgo->format('F d Y H:i:s') . "' | is_future }}", ''],
            'is_leap_year' => ["{{ 'November 2016' | is_leap_year }}", '1'],
            'is_lowercase' => ["{{ 'fhqwhgads' | is_lowercase }}", '1'],
            'is_lowercase fail' => ["{{ 'Sibbie' | is_lowercase }}", ''],
            'is_numberwang who wrote this' => [ "{{ 22 | is_numberwang }}", '1' ],
            'is_numberwang fail why just why' => [ "{{ 100 | is_numberwang }}", '' ],
            'is_numeric' => ["{{ 4815162342 | is_numeric }}", '1'],
            'is_numeric fail' => ["{{ 'just type 4 8 15 16 23 42' | is_numeric }}", ''],
            'is_past' => ["{{ '" . $fiveDaysAgo->format('F d Y H:i:s') . "' | is_past }}", '1'],
            'is_today' => ["{{ '" . $today->format('F d Y H:i:s') . "' | is_today }}", '1'],
            'is_today fail' => ["{{ '" . $fiveDaysAgo->format('F d Y H:i:s') . "' | is_today }}", ''],
            'is_tomorrow' => ["{{ '" . $tomorrow->format('F d Y H:i:s') . "' | is_tomorrow }}", '1'],
            'is_tomorrow fail' => ["{{ '" . $fiveDaysAgo->format('F d Y H:i:s') . "' | is_tomorrow }}", ''],
            'is_uppercase' => ["{{ 'NOISES' | is_uppercase }}", '1'],
            'is_uppercase fail' => ["{{ 'anonymous' | is_uppercase }}", ''],
            'is_url' => ["{{ 'http://google.com/' | is_url }}", '1'],
            'is_url fail' => ["{{ 'waffles' | is_url }}", ''],
            'is_weekday' => ["{{ 'December 25 2015' | is_weekday }}", '1'],
            'is_weekend' => ["{{ 'December 25 2015' | is_weekend }}", ''],
            'is_yesterday' => ["{{ '" . $yesterday->format('F d Y H:i:s') . "' | is_yesterday }}", '1'],
            'is_yesterday fail' => ["{{ '" . $fiveDaysAgo->format('F d Y H:i:s') . "' | is_yesterday }}", ''],
            'iso_format' => ["{{ 'June 19 2020' | iso_format('MMMM Do YYYY, h:mm:ss a') }}", 'June 19th 2020, 12:00:00 am'],
            'lcfirst' => ["{{ 'Wow' | lcfirst }}", 'wow'],
            'md5' => ["{{ 'hello' | md5 }}", '5d41402abc4b2a76b9719d911017c592'],
            'minutes_ago' => ["{{ '" . $fiveMinutesAgo->format('F d Y H:i:s') . "' | minutes_ago }}", '5'],
            'modify_date' => ["{{ '" . $today->format('F d Y H:i:s') . "' | modify_date('-1 day') }}", $yesterday->format('Y-m-d H:i:s')],
            'months_ago' => ["{{ '" . $fiveMonthsAgo->format('Y-m-d H:i:s') . "' | months_ago }}", '5'],
            'obfuscate' => ["{{ 'Abracadabra' | obfuscate }}", '&#x41;b&#114;&#97;&#x63;a&#100;&#x61;br&#x61;'], // No example
            'obfuscate_email' => [ "{{ 'holler@example.com'|obfuscate_email }}", '&#x68;o&#108;&#108;&#x65;r&#64;&#x65;xa&#x6d;ple&#x2e;&#x63;&#111;&#x6d;' ],
            'option_list' => ["{{ ['blog', 'news', 'wigs'] | option_list }}", 'blog|news|wigs'],
            'option_list w/params' => ["{{ ['blog', 'news', 'wigs'] | option_list('!!') }}", 'blog!!news!!wigs'], // No example
            'pathinfo w/ 2 params' => ["{{ '/local/file/example.pdf' | pathinfo('extension') }}", 'pdf'],
            'piped' => ["{{ ['blog', 'news', 'wigs'] | piped }}", 'blog|news|wigs'],
            'rawurlencode' => ["http://example.com/{{ 'please and thank you/Mommy' | rawurlencode }}", 'http://example.com/please%20and%20thank%20you/Mommy'],
            'read_time' => ["{{ 'Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here...' | read_time }} min", '2 min'],
            'read_time w/params' => ["{{ 'Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here... Pretend there are lots of words here...' | read_time(100) }} min", '3 min'],
            'regex_replace' => ["{{ 'This cat video is the best thing ever.' | regex_replace('best', 'okayest') }}", 'This cat video is the okayest thing ever.'],
            'relative' => ["{{ '" . $fiveYearsAgo->format('Y-m-d H:i:s') . "' | relative }}", '5 years ago'],
            'remove_left' => ["{{ '@statamic' | remove_left('@') }}", 'statamic'],
            'remove_right' => ["{{ 'http://statamic.com/' | remove_right('/') }}", 'http://statamic.com'],
            'reverse string' => ["{{ 'repaid' | reverse }}", 'diaper'],
            'safe_truncate' => ["{{ 'So, here’s some advice I wish I woulda got when I was your age: Live every week like it’s Shark Week.' | safe_truncate(87) }}", 'So, here’s some advice I wish I woulda got when I was your age: Live every week like'],
            'safe_truncate w/params' => ["{{ 'So, here’s some advice I wish I woulda got when I was your age: Live every week like it’s Shark Week.' | safe_truncate(90, '...') }}", 'So, here’s some advice I wish I woulda got when I was your age: Live every week like...'],
            'seconds_ago' => ["{{ '" . $fiveSecondsAgo->format('F d Y H:i:s') . "' | seconds_ago }}", '5'],
            'singular' => ["{{ 'nickles' | singular }}", 'nickle'],
            'smartypants' => ["{{ '\"What\'s your favorite album?\" asked Lars. ``...And Justice for All\'\' replied Kirk -- who was icing his hands after a 20 minute guitar solo.' | smartypants }}", '&amp;#8220;What&amp;#8217;s your favorite album?&amp;#8221; asked Lars. &amp;#8220;&amp;#8230;And Justice for All&amp;#8221; replied Kirk &amp;#8212; who was icing his hands after a 20 minute guitar solo.'],
            'sort' => ["{{ ['Zebra', 'Alpha', 'Bravo'] | sort }}", '["Alpha","Bravo","Zebra"]'],
            'sort w/params' => ["{{ [ { last_name: 'Zebra', first_name: 'Zealous' }, { last_name: 'Alpha', first_name: 'Altruistic' }, { last_name: 'Bravo', first_name: 'Blathering' }] | sort('last_name') }}", '[{"last_name":"Alpha","first_name":"Altruistic"},{"last_name":"Bravo","first_name":"Blathering"},{"last_name":"Zebra","first_name":"Zealous"}]'],
            'spaceless' => ["{{ '   <p>I copy & pasted        <a href=\"http://goodnightchrome.show\">this link</a>   <strong>for you!</strong>    </p>' | spaceless }}", ' &lt;p&gt;I copy &amp; pasted &lt;a href=&quot;http://goodnightchrome.show&quot;&gt;this link&lt;/a&gt;&lt;strong&gt;for you!&lt;/strong&gt;&lt;/p&gt;'],
            'starts_with' => ["{{ 'Actually, I disagree because this is the internet.' | starts_with('actually') }}", '1'],
            'starts_with fail' => ["{{ 'Actually, I disagree because this is the internet.' | starts_with('respectfully') }}", ''],
            'sum' => ["{{ [5, 10, 20, 40] | sum }}", '75'],
            'sum w/params' => ["{{  [ { player: 'Luke Skywalker', score: 750 }, { player: 'Wedge Antilles', score: 688 }, { player: 'Jar Jar Binks', score: 1425 }] | sum('score') }}", '2863'],
            'surround' => ["{{ '_' | surround('o') }}", 'o_o'],
            'swap_case' => ["{{ 'IpHONE' | swap_case }}", 'iPhone'],
            'table' => [ "{{ [ {cells: ['bacon', 'sushi']}, {cells: ['broccoli', 'kale']} ] | table }}", '<table><tr><td>bacon</td><td>sushi</td></tr><tr><td>broccoli</td><td>kale</td></tr></table>' ],
            'tidy' => ["{{ '“I see…”' | tidy }}", '&quot;I see...&quot;'],
            'title' => ["{{ 'It was one of the best adventures of my life' | title }}", 'It Was One of the Best Adventures of My Life'],
            'trim' => ["{{  '    This is so sloppy   ' | trim }}", 'This is so sloppy'],
            'truncate' => ["{{ 'So, here’s some advice I wish I woulda got when I was your age: Live every week like it’s Shark Week.' | truncate(86) }}", 'So, here’s some advice I wish I woulda got when I was your age: Live every week like i'],
            'truncate w/params' => ["{{ 'So, here’s some advice I wish I woulda got when I was your age: Live every week like it’s Shark Week.' | truncate(89, '...') }}", 'So, here’s some advice I wish I woulda got when I was your age: Live every week like i...'],
            'ucfirst' => ["{{ 'i wanna go home.' | ucfirst }}", 'I wanna go home.'],
            'ul' => [ "{{ [ 'sushi', 'broccoli', 'kale' ] | ul }}", '<ul><li>sushi</li><li>broccoli</li><li>kale</li></ul>' ],
            'underscored' => ["{{ 'Please and thank you' | underscored }}", 'please_and_thank_you'],
            'unique' => [ "{{ [ 'zebra', 'hippo', 'hyena', 'giraffe', 'zebra', 'hippo', 'hippo', 'hippo', 'hippo' ] | unique | join(', ') }}", 'zebra, hippo, hyena, giraffe' ],
            'upper' => ["{{ 'That is über neat.' | upper }}", 'THAT IS ÜBER NEAT.'],
            'urldecode' => ["{{ 'I+just+want+%26+need+%24pecial+characters%21' | urldecode }}", 'I just want & need $pecial characters!'],
            'urlencode' => ["{{ 'I just want &amp; need \$pecial characters!' | urlencode }}", 'I+just+want+%26amp%3B+need+%24pecial+characters%21'],
            'weeks_ago' => ["{{ '" . $fiveWeeksAgo->format('F d Y H:i:s') . "' | weeks_ago }}", '5'],
            'widont' => ["{{ 'I Just Want Pretty Headlines and Sentences' | widont }}", 'I Just Want Pretty Headlines and&amp;nbsp;Sentences'],
            'widont w/params' => ["{{ 'I Just Want Pretty Headlines and Sentences' | widont(4) }}", 'I Just Want&amp;nbsp;Pretty&amp;nbsp;Headlines&amp;nbsp;and&amp;nbsp;Sentences'],
            'word_count' => [ "{{ 'There are probably seven words in this sentence.' | word_count }}", '8' ],
            'wrap' => ["{{ 'As the World Turns' | wrap('h1') }}", '&lt;h1&gt;As the World Turns&lt;/h1&gt;'],
            'wrap w/params' => ["{{ 'As the World Turns' | wrap('h1.fast.furious') }}", '&lt;h1 class=&quot;fast furious&quot;&gt;As the World Turns&lt;/h1&gt;'],
            'years_ago' => ["{{ '" . $fiveYearsAgo->format('F d Y H:i:s') . "' | years_ago }}", '5'],
        ];
    }
}
