<?php
class SluggedBehavior extends ModelBehavior {

/**
 * Contains configuration settings for use with individual model objects. This
 * is used because if multiple models use this Behavior, each will use the same
 * object instance. Individual model settings should be stored as an
 * associative array, keyed off of the model name.
 *
 * @var array
 */
	protected $_defaultSettings = array(
		'label' => null,
		'slugField' => 'slug',
		'replace' => array(
			'&' => 'and',
			'+' => 'and',
			' ' => '-',
			'\'' => ''
		),
		'unique' => false,
		'routeView' => 'view',
		'routeParams' => [
			':primaryKey',
			':slug'
		]
	);

/**
 * Setup this behavior with the specified configuration settings.
 *
 * @param Model $model Model using this behavior
 * @param array $config Configuration settings for $model
 * @return void
 */
	public function setup(Model $Model, $config = []) {
		$defaults = array(
			'label' => $Model->displayField
		);
		$defaults += $this->_defaultSettings;
		$config += $defaults;
		$this->settings[$Model->alias] = $config;
		return;
	}

/**
 * Called during validation operations, before validation. Please note that custom
 * validation rules can be defined in $validate.
 *
 * @param array $options Options passed from Model::save().
 * @return bool True if validate operation should continue, false to abort
 */
	public function beforeValidate(Model $Model, $options = []) {
		if (
			array_key_exists($this->settings[$Model->alias]['slugField'], $Model->data[$Model->alias])
			&& empty($Model->data[$Model->alias][$this->settings[$Model->alias]['slugField']])
		) {
			$Model->data[$Model->alias][$this->settings[$Model->alias]['slugField']] = $this->generateSlug($Model);
		}

		return true;
	}

/**
 * Called after each find operation. Uses the defined settings and appends a
 * sluggable route to each of the primary model results
 *
 * @param mixed $results The results of the find operation
 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
 * @return mixed Result of the find operation
 */
	public function afterFind(Model $Model, $results, $primary = false) {
		if (empty($this->settings[$Model->alias]['routeParams']) || $primary === false) {
			return $results;
		}

		if (!empty($this->settings[$Model->alias]['plugin'])) {
			$pluginPath = $this->settings[$Model->alias]['plugin'];
		} else {
			$pluginPath = Inflector::underscore($Model->plugin);
		}

		if (!empty($this->settings[$Model->alias]['controller'])) {
			$controllerPath = $this->settings[$Model->alias]['controller'];
		} else {
			$controllerPath = Inflector::underscore(Inflector::pluralize($Model->alias));
		}

		foreach ($results as &$result) {
			if (!empty($result[$Model->alias][$this->settings[$Model->alias]['slugField']])) {
				$params = array_map(function ($value) use ($result, $Model) {
					$value = str_replace(
						[':primaryKey', ':slug'],
						[
							$result[$Model->alias][$Model->primaryKey],
							$result[$Model->alias][$this->settings[$Model->alias]['slugField']]
						],
						$value
					);
					if (preg_match('/:(\w+)/', $value, $matches) === 1 && !empty($result[$Model->alias][$matches[1]])) {
						$value = str_replace(':' . $matches[1], $result[$Model->alias][$matches[1]], $value);
					}
					return $value;
				}, $this->settings[$Model->alias]['routeParams']);

				$route = [
					'admin' => false,
					'plugin' => Inflector::underscore($Model->plugin),
					'controller' => $controllerPath,
					'action' => $this->settings[$Model->alias]['routeView']
				] + $params;

				$result[$Model->alias]['slugged_route'] = $route;
			}
		}

		return $results;
	}

/**
 * Constructs the slug from the model data based from the supplied label in the settings
 *
 * @param Model $model Model using this behavior
 * @return string
 */
	public function generateSlug(Model $Model) {
		$slug = str_replace(
			array_keys($this->settings[$Model->alias]['replace']),
			$this->settings[$Model->alias]['replace'],
			$Model->data[$Model->alias][$this->settings[$Model->alias]['label']]
		);

		$slug = Inflector::slug(strtolower($slug), '-');

		if ($this->settings[$Model->alias]['unique'] === false) {
			return $slug;
		}

		$suffix = 0;

		do {
			$alias = $suffix === 0 ? $slug : $slug . '-' . $suffix;
			$found = $Model->find('count', [
				'conditions' => [
					$Model->alias . '.' . $this->settings[$Model->alias]['slugField'] => $alias,
					$Model->alias . '.' . $Model->primaryKey . ' <>' => $Model->id,
				]
			]);
			++$suffix;
		} while ($found);

		return $alias;
	}

/**
 * Returns the model data relating to the supplied slug string
 *
 * @param Model $model Model using this behavior
 * @param str $slug The string containing the string to query against
 * @param int $id The unique id of the queired item
 * @return array
 */
	public function findBySlug(Model $Model, $slug, $id = null) {
		$params = array(
			'conditions' => array(
				$Model->alias . '.' . $this->settings[$Model->alias]['slugField'] => $slug
			)
		);
		if (!empty($id)) {
			$params['conditions'][$Model->alias . '.' . $Model->primaryKey] = $id;
		}
		return $Model->find('first', $params);
	}

}
