<?php

App::uses('AppShell', 'Console/Command');
App::uses('EvClassRegistry', 'EvCore.Lib');
class PopulateSlugFieldShell extends AppShell {

	public function main() {
		if (empty($this->args)) {
			$this->out('<error>Provide the model that you wish to build the slugs for</error>');
			die;
		}

		$Model = EvClassRegistry::init($this->args[0]);

		if (!array_key_exists('Slugged', $Model->actsAs)) {
			$this->out('<error>Provided model doesn\'t have the Sluggable behaviour</error>');
			die;
		}

		//Get the current behavior settings from the behavior for the current model.
		$sluggableConf = $Model->Behaviors->Slugged->settings[$Model->alias];

		if (!$Model->hasField($sluggableConf['slugField'])) {
			$this->out('<error>Model doesn\'nt have the field . ' . $sluggableConf['slugField'] . '</error>');
			exit;
		}

		$this->out('<info>Disabling all behaviours except Sluggable</info>');
		foreach ($Model->actsAs as $behavior => $config) {
			// We don't want to disable the sluggable behaviour.
			if ($behavior === 'Slugged') {
				continue;
			}
			$Model->Behaviors->unload($behavior);
			$Model->Behaviors->disable($behavior);
		}

		$this->out('<info>Saving model data...</info>');

		$data = $this->__getModelData($Model, $sluggableConf);

		$dataNotSaved = [];
		while (!empty($data)) {
			foreach ($data as $item) {
				//Disable validation
				$originalValidation = $Model->validate;
				$Model->validate = [];

				$Model->clear();
				$saved = $Model->save($item);

				if (!$saved) {
					$itemId = $item[$Model->alias][$Model->primaryKey];
					$dataNotSaved[] = $itemId;

					$this->out('<warning>Could not save ' . $Model->alias . ' #' . $itemId . '</warning>');
				}

				$Model->validate = $originalValidation;
			}

			$data = $this->__getModelData($Model, $sluggableConf, $dataNotSaved);
		}

		$this->out('<info>Model data saved...</info>');

		foreach ($Model->actsAs as $behavior => $config) {
			// It's already enabled so no need.
			if ($behavior === 'Slugged') {
				continue;
			}
			$Model->Behaviors->enable($behavior);
			$Model->Behaviors->load($behavior);
		}
		$this->out('<info>All behaviours re-enabled...</info>');
	}

/**
 * Get data to update sluggable fields on.
 *
 * @param obj   $Model         Object model with sluggable behavior.
 * @param array $sluggableConf The current sluggable config.
 * @param array $ignoreItemIds Ignore items with an id in this array. Used to stop items being updated.
 * @return array Data found to be slugged.
 */
	private function __getModelData($Model, $sluggableConf, $ignoreItemIds = []) {
		return $Model->find(
			'all',
			[
				'conditions' => [
					$Model->alias . '.' . $Model->primaryKey . ' !=' => $ignoreItemIds,
					'OR' => [
						$Model->alias . '.' . $sluggableConf['slugField'] => '',
						$Model->alias . '.' . $sluggableConf['slugField'] . ' IS NULL',
					],
				],
				'limit' => 100,
			]
		);
	}
}
