<?php

class SitemapBehavior extends ModelBehavior {

	public $settings = [];

/**
 * Setup this behavior with the specified configuration settings.
 *
 * @param Model $Model Model using this behavior
 * @param array $settings Configuration settings for $model
 * @return void
 */
	public function setup(Model $Model, $settings = []) {
		if (!isset($this->settings[$Model->alias])) {
			// assemble based on default params defined in config
			$this->settings[$Model->alias] = Configure::read('EvSitemap.defaults');
		}

		$this->settings[$Model->alias] = array_merge($this->settings[$Model->alias], $settings);
	}

/**
 * Get the SitemapConditions if available
 *
 * @param Model $Model The model
 * @param array $results Array of results
 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
 * @return mixed
 */
	public function afterFind(Model $Model, $results, $primary = false) {
		if ($primary !== true) {
			return $results;
		}
		$SitemapConditions = EvClassRegistry::init('EvSitemap.SitemapConditions');
		foreach ($results as &$result) {
			if (empty($result[$Model->alias]['id'])) {
				continue;
			}
			$resultConditions = $SitemapConditions->find('first', [
				'conditions' => [
					'model' => $Model->alias,
					'model_id' => $result[$Model->alias]['id']
				]
			]);
			if (!empty($resultConditions)) {
				$result['SitemapConditions'] = reset($resultConditions);
			}
		}
		return $results;
	}

/**
 * Save the meta data
 *
 * @see ModelBehavior::afterSave()
 */
	public function afterSave(Model $Model, $created, $options = array()) {
		parent::afterSave($Model, $created, $options);

		if (CakePlugin::loaded('Queue')) {
			$QueuedTask = EvClassRegistry::init('Queue.QueuedTask');

			$existingJob = $QueuedTask->find(
				'first',
				[
					'conditions' => [
						$QueuedTask->alias . '.jobType' => 'RefreshSitemapListings',
						$QueuedTask->alias . '.fetched' => null,
					]
				]
			);

			if (empty($existingJob)) {
				$QueuedTask->createJob('RefreshSitemapListings');
			}
		}

		if (empty($Model->data[$Model->alias]['SitemapConditions'])) {
			return;
		}

		$data = $Model->data[$Model->alias];
		$data['SitemapConditions']['model'] = $Model->alias;
		$data['SitemapConditions']['model_id'] = $Model->data[$Model->alias]['id'];

		$SitemapConditions = EvClassRegistry::init('EvSitemap.SitemapConditions');
		if (empty($data['SitemapConditions']['id'])) {
			$SitemapConditions->create();
		}
		$SitemapConditions->save($data);
	}

/**
 * Get a list of sitemap data for the model
 *
 * @param Model $Model Model using this behavior
 * @return array
 */
	public function getSitemapData(Model $Model) {
		$modelData = $Model->find('all', [
			'conditions' => $this->_getConditions($Model),
			'joins' => [
				[
					'table' => 'ev_sitemap_sitemap_conditions',
					'alias' => 'SitemapConditions',
					'type' => 'LEFT',
					'conditions' => [
						'SitemapConditions.model' => $Model->alias,
						'SitemapConditions.model_id = ' . $Model->alias . '.id',
					],
				]
			],
			'recursive' => -1,
		]);

		return $this->_assembleSitemap($Model, $modelData);
	}

/**
 * Set 'is_active' as default, merge with the Model's conditions
 *
 * @param Model $Model Model using this behavior
 * @return array
 */
	protected function _getConditions(Model $Model) {
		$defaultConditions = [
			[
				'OR' => [
					'SitemapConditions.noindex' => false,
					// Not every model entry will have a SitemapConditions row, assume they are to be included
					'SitemapConditions.id IS NULL',
				],
			],
		];
		if ($Model->hasField('is_active')) {
			$defaultConditions['is_active'] = true;
		}
		return array_merge($defaultConditions, $this->settings[$Model->alias]['conditions']);
	}

/**
 * Format the sitemap data
 *
 * @param Model $Model Model using this behavior
 * @param array $modelData Data from the model
 * @return array
 */
	protected function _assembleSitemap(Model $Model, $modelData = []) {
		$data = [];

		if (empty($modelData)) {
			return $data;
		}

		foreach ($modelData as $key => $item) {

			if (
				in_array(
					$item[$Model->alias][$this->settings[$Model->alias]['primaryKey']],
					$this->settings[$Model->alias]['ignoreList']
				)
			) {

				continue;
			}

			$data[$key] = [];

			$pluginName = null;
			$controllerName = Inflector::tableize($Model->name);
			$controllerAction = $this->settings[$Model->alias]['controllerAction'];

			//If the controller has been overridden and a plugin has been provided, set it here
			if (isset($Model->actsAs['Sitemap']['plugin'])) {
				$pluginName = $Model->actsAs['Sitemap']['plugin'];
			}

			if (isset($Model->actsAs['Sitemap']['controller'])) {
				$controllerName = $Model->actsAs['Sitemap']['controller'];
			}

			if (isset($Model->actsAs['Sitemap']['controllerAction'])) {
				$controllerAction = $Model->actsAs['Sitemap']['controllerAction'];
			}

			// URL used for the sitemap is generated from its Route by default, but will populate from a custom function if available
			if (method_exists($Model, $this->settings[$Model->alias]['loc']) && is_callable(array($Model, $this->settings[$Model->alias]['loc']))) {
				$data[$key]['loc'] = call_user_func(array($Model, $this->settings[$Model->alias]['loc']), $item);
			} else {
				$data[$key]['loc'] = Router::url(array(
					'plugin' => $pluginName,
					'controller' => $controllerName,
					'action' => $controllerAction,
					$item[$Model->alias][$this->settings[$Model->alias]['primaryKey']]
				), true);
			}

			if ($this->settings[$Model->alias]['lastmod'] !== false) {
				$data[$key]['lastmod'] = $item[$Model->alias][$this->settings[$Model->alias]['lastmod']];
			}

			if ($this->settings[$Model->alias]['changefreq'] !== false) {
				$data[$key]['changefreq'] = $this->settings[$Model->alias]['changefreq'];
			}

			if (is_numeric(($this->settings[$Model->alias]['readPriority']))) {
				$data[$key]['readPriority'] = $this->settings[$Model->alias]['readPriority'];
			} elseif (!empty($item[$Model->alias][$this->settings[$Model->alias]['readPriority']])) {
				$data[$key]['readPriority'] = $item[$Model->alias][$this->settings[$Model->alias]['readPriority']];
			}

			// include the item name / title for the website version
			if (isset($Model->displayField) && isset($item[$Model->alias][$Model->displayField])) {
				$data[$key]['name'] = $item[$Model->alias][$Model->displayField];
			}
		}

		return $data;
	}

}
