<?php

class SitemapLib {

/**
 * Get the sitemap listing. If the cache is enabled then the cached version will
 * attempted to be returned, if not then the sitemap will be generated.
 *
 * @param string $sitemapType The sitemap type, html or xml.
 * @return array.
 */
	public function getListing($sitemapType = 'html') {
		$cacheName = Configure::read('EvSitemap.cacheName');
		$listing = null;

		// If cacheName is defined try get the listing from the cache
		if (!empty($cacheName)) {
			$listing = Cache::read('EvSitemap.listing.' . $sitemapType, $cacheName);
		}

		if (empty($listing)) {
			$listing = $this->generateListing($sitemapType);
		}

		return $listing;
	}

/**
 * Generate the listing for the sitemap. All models are read and any available
 * data will be added to the listing. If the cache is enabled then the listing
 * will be written to the cache.
 *
 * @param string $sitemapType The sitemap type, html or xml.
 * @param bool   $clearCache Clear the existing cache. If a cache already exists
 *                           and this is not set to true then a new listing will
 *                           not be generated.
 * @return array.
 */
	public function generateListing($sitemapType = 'html', $clearCache = false) {
		$cacheName = Configure::read('EvSitemap.cacheName');

		if (! empty(Cache::read('EvSitemap.listing.' . $sitemapType, $cacheName))
			&& !$clearCache
		) {
			return;
		}

		// assign list of models to be used within the sitemap
		$modelListing = $this->_getModelListing();

		if (! empty($modelListing)) {
			foreach ($modelListing as $modelName) {

				list(, $model) = pluginSplit($modelName);

				try {

					$Model = EvClassRegistry::init(ucwords($modelName));

				} catch(\Exception $e) {
					continue;
				}

				if (!empty($Model->actsAs) && $Model->hasBehavior('Sitemap')) {
					$key = ! empty($Model->actsAs['Sitemap']['listingTitle'])
						? $Model->actsAs['Sitemap']['listingTitle']
						: Inflector::humanize(Inflector::underscore(Inflector::pluralize($Model->name)));

					$listing[$key] = $Model->getSitemapData();
				}

				unset($Model);
			}
		}

		// Write back to the cache if defined
		if (!empty($cacheName)) {
			Cache::write('EvSitemap.listing.' . $sitemapType, $listing, $cacheName);
		}

		return $listing;
	}

/**
 * Generate sitemap listings for all sitemap types.
 *
 * @param bool $clearCache Clear the existing cache. If a cache already exists
 *                         and this is not set to true then a new listing will
 *                         not be generated.
 * @return array.
 */
	public function generateListings($clearCache = false) {
		$listings = [];

		$listings['html'] = $this->generateListing('html', $clearCache);
		$listings['xml'] = $this->generateListing('xml', $clearCache);

		return $listings;
	}

/**
 * Clear the cache of one or all sitemap types. This does not regenerate the
 * cache that is cleared.
 *
 * @param ?string $sitemapType The sitemap type, html or xml. Null to clear all
 *                             types. Default null.
 * @return void.
 */
	public function clearCache($sitemapType = null) {
		$cacheName = Configure::read('EvSitemap.cacheName');

		if (empty($cacheName)) {
			return;
		}

		if ($sitemapType === 'html' || $sitemapType === null) {
			Cache::delete('EvSitemap.listing.html', $cacheName);
		}

		if ($sitemapType === 'xml' || $sitemapType === null) {
			Cache::delete('EvSitemap.listing.xml', $cacheName);
		}
	}

/**
 * Clear and regenerate the cache for one or all sitemap types.
 *
 * @param ?string $sitemapType The sitemap type, html or xml. Null to clear all
 *                             types. Default null.
 * @return void.
 */
	public function refreshCache($sitemapType = null) {
		$cacheName = Configure::read('EvSitemap.cacheName');

		if (empty($cacheName)) {
			return;
		}

		$this->clearCache($sitemapType);

		if ($sitemapType === null) {
			$this->generateListings(true);
			return;
		}

		$this->generateListing($sitemapType, true);
	}

/**
 * Fetches the list of models to check
 *
 * @return array A list of models
 */
	protected function _getModelListing() {
		$listing = array();

		$ignoreList = Configure::read('EvSitemap.modelIgnoreList');

		// Get all the models in the main app model directory
		$modelList = App::objects('Model');

		// get all the plugins so we can get all their models
		$pluginList = App::objects('plugins');

		// loop and get the models
		foreach ($pluginList as $plugin) {

			// get the plugin models and prefix with the plugin name
			$pluginModel = App::objects($plugin . '.Model');

			foreach ($pluginModel as $key => $model) {

				$pluginModel[$key] = $plugin . '.' . $model;
			}

			$modelList = array_merge($modelList, $pluginModel);
		}

		if (! empty($modelList)) {

			foreach ($modelList as $modelName) {

				if (in_array($modelName, $ignoreList)) {

					continue;
				}

				$listing[] = $modelName;
			}
		}

		return $listing;
	}
}
