<?php

App::uses('BaseMigrationModelUtil', 'EvMigrationUtil.Lib');
App::uses('MigrationUtil', 'EvMigrationUtil.Lib');
App::uses('InflectorExt', 'EvInflector.Lib');

class SiteSettingMigrationModelUtil extends BaseMigrationModelUtil {

/**
 * Builds a Site Setting. Will also add the category if it doesn't exist. Uses the humanized version of the slug as the name.
 * expects an array with setting and category keys
 *
 * @param string $slug The setting slug
 * @param string $categorySlug The category slug to place this setting in. The category will be created if it doesn't exist
 * @param string $type The field type
 * @param array $options Options to override the default options. Options include name, valueOptions, value, sequence, preventEdit, categoryName, categorySequence
 *
 * @return int|bool The site setting id or false on fail
 */
	public function createOnceWithCategory($slug, $categorySlug, $type, $options = []) {
		$options = array_merge([
			'name' => InflectorExt::humanize($slug),
			'valueOptions' => '',
			'value' => null,
			'sequence' => 0,
			'preventEdit' => false,
			'categoryName' => InflectorExt::humanize($categorySlug),
			'categorySequence' => 0,
		], $options);

		$categoryUtil = MigrationUtil::init('EvSiteSettings.SiteSettingCategory');
		$categoryId = $categoryUtil->createCategoryOnce($categorySlug, $options['categoryName'], $options['categorySequence'], $options['preventEdit']);

		if (empty($categoryId)) {
			return false;
		}

		return $this->createOnce([
			$this->Model->alias . '.array_slug' => $slug,
			$this->Model->alias . '.site_setting_category_id' => $categoryId,
		], [
			'SiteSetting' => [
				'array_slug' => $slug,
				'name' => $options['name'],
				'type' => $type,
				'options' => $options['valueOptions'],
				'value' => $options['value'],
				'sequence' => $options['sequence'],
				'site_setting_category_id' => $categoryId,
				'prevent_edit' => $options['preventEdit']
			]
		]);
	}

/**
 * Update a site setting in a certain site setting category
 *
 * @param string $categorySlug the slug of the site setting category
 *
 * @param string $settingSlug the slug of the setting to update
 *
 * @param array $values the values to update
 *
 * @return bool|int
 */
	public function updateSingleWithCategory($categorySlug, $settingSlug, $values) {
		$categoryId = $this->Model->SiteSettingCategory->field('id', ['array_slug' => $categorySlug]);
		return $this->updateSingle([
			'site_setting_category_id' => $categoryId,
			'array_slug' => $settingSlug
		], $values);
	}

/**
 * Deletes based on a slug
 *
 * @param string $slug         The setting slug
 * @param string $categorySlug The category slug
 * @return True on success, false if no category found or could not be deleted.
 */
	public function deleteWithSlug($slug, $categorySlug) {
		$categoryId = $this->Model->SiteSettingCategory->field('id', ['array_slug' => $categorySlug]);
		if (!empty($categoryId)) {
			return $this->deleteAll([
				$this->Model->alias . '.array_slug' => $slug,
				$this->Model->alias . '.site_setting_category_id' => $categoryId
			]);
		}

		//Couldn't find category to remove site setting from.
		return false;
	}

/**
 * Exports a page and creates a migration
 *
 * @param string $identifier The slugs passed as category.setting
 * @return bool True on success
 * @throws  NotFoundException if the site setting is not found
 */
	public function getExportMigration($identifier) {
		list($categorySlug, $settingSlug) = explode('.', $identifier);

		$migrationUp = PHP_EOL;
		$migrationDown = PHP_EOL;
		$migration = '';

		$siteSetting = $this->Model->find('first', [
			'joins' => [
				[
					'table' => 'ev_site_settings_site_setting_categories',
					'alias' => 'Category',
					'type' => 'INNER',
					'conditions' => [
						'Category.id = SiteSetting.site_setting_category_id',
						'Category.array_slug' => $categorySlug,
					]
				]
			],
			'conditions' => [
				'SiteSetting.array_slug' => $settingSlug,
			],
			'fields' => [
				'*'
			]
		]);

		if (empty($siteSetting)) {
			throw new NotFoundException();
		}

		//Common
		$migration .= '$SiteSettingUtil = MigrationUtil::init(\'EvSiteSettings.SiteSetting\');' . PHP_EOL;

		// Up
		$migrationUp .= '$SiteSettingUtil->createOnceWithCategory(\'' . $settingSlug . '\', \'' . $categorySlug . '\', \'' . $siteSetting['SiteSetting']['type'] . '\', [' . PHP_EOL;
		$params = [
			'name' => $siteSetting['SiteSetting']['name'],
			'valueOptions' => $siteSetting['SiteSetting']['options'],
			'value' => $siteSetting['SiteSetting']['value'],
			'sequence' => $siteSetting['SiteSetting']['sequence'],
			'preventEdit' => $siteSetting['SiteSetting']['prevent_edit'],
			'categoryName' => $siteSetting['Category']['name'],
			'categorySequence' => $siteSetting['Category']['sequence'],
		];
		foreach ($params as $field => $value) {
			$migrationUp .= '	\'' . $field . '\' => ' . $this->_migrationValue($value) . ',' . PHP_EOL;
		}
		$migrationUp .= ']);' . PHP_EOL;

		// Down
		// For the down only clear the setting not the category in case settings have been created in this category
		$migrationDown .= '$SiteSettingUtil->deleteWithSlug(\'' . $settingSlug . '\', \'' . $categorySlug . '\');';

		return [
			'common' => $migration,
			'up' => $migrationUp,
			'down' => $migrationDown
		];
	}

}
