<?php

App::uses('EvSiteSettingsAppController', 'EvSiteSettings.Controller');
App::uses('ArrayUtil', 'EvCore.Lib');

/**
 * Handles requests for site settings
 */
class SiteSettingsController extends EvSiteSettingsAppController {

	public function beforeFilter() {
		$this->adminActions[] = 'admin_manage';

		parent::beforeFilter();
	}

	/**
	 * Defines the buttons displayed on the admin toolbar
	 */
	protected function _adminIndexToolbar() {
		if ($this->Permissions->hasSuperAdminPermission($this->Auth->user())) {

			$toolbar = array(
				'Add New Setting' => array(
					'url' => array(
						'action' => 'add'
					)
				),
				'Categories' => array(
					'class' => 'toolbar__btn--folder',
					'url' => array(
						'controller' => 'site_setting_categories',
						'action' => 'index'
					)
				)
			);

			if ($this->request->params['action'] == 'admin_index') {

				$toolbar['Manage Settings'] = array(
					'class' => 'toolbar__btn--reorder',
					'url' => array(
						'action' => 'manage'
					)
				);

			} else {

				$toolbar['View Settings'] = array(
					'class' => 'toolbar__btn--copy',
					'url' => array(
						'action' => 'index'
					)
				);
			}

			return $toolbar;
		} else {

			return;
		}
	}

	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};
		$siteSettingCategories = $Model->SiteSettingCategory->find('list');

		$types = array(
			'string' => 'Input Box',
			'text_plain' => 'Textarea',
			'select' => 'Select',
			'html_lite' => 'WYSIWYG Lite',
			'html' => 'WYSIWYG',
			'boolean' => 'Checkbox',
			'date' => 'Date Picker',
			'datetime' => 'Date Time Picker',
			'map' => 'Map'
		);

		$this->set(
			compact('siteSettingCategories', 'types')
		);
	}

	/**
	 * redefine to set value to plain to prevent p tags around everything
	 */
	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$fields['SiteSetting.type']['type'] = 'select';
		$fields['SiteSetting.value']['type'] = 'text_plain';
		$fields['SiteSetting.options']['type'] = 'text_plain';
		$fields['SiteSetting.options']['displayInfo'] = '<b>Format:</b> key:value,key2:value2,key3:value3';

		$fields = ArrayUtil::addAfter(
			$fields,
			'SiteSetting.site_setting_category_id',
			array(
				'SiteSetting.sequence' => array(
					'type' => 'string',
					'label' => 'Sequence'
				)
			)
		);

		return $fields;
	}

	/**
	 * show the array slug on the index page
	 */
	protected function _adminIndexColumnsWhitelist() {
		$modelAlias = $this->{$this->modelClass}->alias;

		$fields = parent::_adminIndexColumnsWhitelist();

		$fields[] = $modelAlias . '.array_slug';

		return $fields;
	}

	/**
	 * show the setting category on the listing page
	 *
	 */
	protected function _adminIndexColumns() {
		$columns = parent::_adminIndexColumns();

		$columns = ArrayUtil::addAfter(
			$columns,
			'SiteSetting.array_slug',
			array(
				'SiteSettingCategory.name' => array(
					'type' => 'string',
					'label' => 'Category'
				)
			)
		);

		return $columns;
	}

	/**
	 * contain the categories
	 */
	protected function _adminIndexPaginate() {
		$paginate = parent::_adminIndexPaginate();

		$paginate['contain'][] = 'SiteSettingCategory';

		return $paginate;
	}

	/**
	 * Landing Page - Edit Site settings
	 *
	 * Site settings are handled differently from other models
	 * This screen is a combined index & edit
	 */
	public function admin_index() {
		$fields = array();

		$data = $this->SiteSetting->find('grouped', array(
			'order' => array(
				'SiteSettingCategory.sequence' => 'ASC',
				'SiteSetting.sequence' => 'ASC',
				'SiteSetting.array_slug' => 'ASC'
			)
		));

		$fields = $this->_buildIndexFields($fields, $data);

		// handle a form post
		if (!empty($this->request->data)) {

			foreach ($this->request->data['SiteSetting'] as $fieldName => $value) {
				$siteSettingFields = array('value' => '"' . addslashes($value) . '"');
				$siteSettingConditions = array('SiteSetting.array_slug' => $fieldName);
				$this->SiteSetting->updateAll($siteSettingFields, $siteSettingConditions);
			}

			$this->Session->setFlash(
				array(
					'title' => 'Settings',
					'description' => 'Settings have been saved'
				),
				'flash_success'
			);

		} else {

			foreach ($data as $category) {
				foreach ($category as $setting) {
					$this->request->data['SiteSetting'][$setting['SiteSetting']['array_slug']] = $setting['SiteSetting']['value'];
				}
			}
		}

		$this->set('fields', $fields);
		$this->set('toolbar', $this->_adminIndexToolbar());
		$this->set('title_for_layout', 'Site Settings');
		$this->view = 'EvSiteSettings.SiteSettings/admin_index';
	}

	protected function _buildIndexFields(&$fields, $data) {
		foreach ($data as $categoryName => $category) {
			$this->_buildIndexSiteSettingCategoryFields($fields, $categoryName, $category);
		}

		return $fields;
	}

	protected function _buildIndexSiteSettingCategoryFields(&$fields, $categoryName, $category) {
		foreach ($category as $setting) {
			$this->_buildIndexSiteSettingField($fields, $categoryName, $category, $setting);
		}

		return $fields;
	}

	protected function _buildIndexSiteSettingField(&$fields, $categoryName, $category, $setting) {
		$preventEditCategory = (isset($setting['SiteSettingCategory']['prevent_edit']) && intval($setting['SiteSettingCategory']['prevent_edit']) === 1);
		$preventEditSetting = (isset($setting['SiteSetting']['prevent_edit']) && intval($setting['SiteSetting']['prevent_edit']) === 1);

		if ($preventEditCategory || $preventEditSetting) {
			return;
		}

		$fields[$categoryName]['SiteSetting.' . $setting['SiteSetting']['array_slug']] = array(
			'type' => $setting['SiteSetting']['type'],
			'label' => $setting['SiteSetting']['name'] != "" ? $setting['SiteSetting']['name'] : $setting['SiteSetting']['array_slug']
		);

		if ($this->Permissions->hasSuperAdminPermission($this->Auth->user())) {

			$fields[$categoryName]['SiteSetting.' . $setting['SiteSetting']['array_slug']]['displayInfo'] =
				'Configure Key: <b>SiteSetting.' . $setting['SiteSettingCategory']['array_slug'] . '.' . $setting['SiteSetting']['array_slug'] . '</b>';
		}

		if (! empty($setting['SiteSetting']['options'])) {

			$fieldOptions = array();
			$options = explode(',', $setting['SiteSetting']['options']);

			foreach ($options as $option) {

				if (strpos($option, ':') !== false) {
					list($key, $value) = explode(':', $option, 2);
				} else {
					$key = $value = $option;
				}

				$fieldOptions[$key] = $value;
			}

			$fields[$categoryName]['SiteSetting.' . $setting['SiteSetting']['array_slug']]['options'] = $fieldOptions;
		}

		return $fields;
	}

	/**
	 * An Evoluted Admin only manage page to easily, edit / delete any settings
	 */
	public function admin_manage() {
		parent::admin_index();
	}
}
