<?php

App::uses('AppShell', 'Console/Command');

App::uses('Controller', 'Controller');
App::uses('ComponentCollection', 'Controller');
App::uses('AclComponent', 'Controller/Component');

class EvSiteSettingsPermissionsShell extends AppShell {

/**
 * Root node name.
 *
 * @var string
 **/
	public $rootNode = 'site_settings';

/**
 * Internal Clean Actions switch
 *
 * @var boolean
 **/
	protected $_clean = false;

/**
 * Start up And load Acl Component / Aco model
 *
 * @return void
 **/
	public function startup($controller = null) {
		if (!$controller) {
			$controller = new Controller(new CakeRequest());
		}
		$collection = new ComponentCollection();
		$this->Acl = new AclComponent($collection);
		$this->Acl->startup($controller);
		$this->Aco = $this->Acl->Aco;
		$this->controller = $controller;

		$this->SiteSettingCategory = EvClassRegistry::init('EvSiteSettings.SiteSettingCategory');
		$this->SiteSetting = EvClassRegistry::init('EvSiteSettings.SiteSetting');
	}

/**
 * Sync the ACO table
 *
 * @return void
 **/
	public function aco_sync($params = array()) {
		$this->_clean = true;
		$this->aco_update($params);
	}

/**
 * Updates the Aco Tree with new site setting categories and site settings.
 *
 * @return void
 **/
	public function aco_update($params = array()) {
		$root = $this->_checkNode($this->rootNode, $this->rootNode, null);

		$categories = $this->getSiteSettingCategoryList();
		$this->_updateSiteSettingCategories($root, $categories);
		$this->out(__('<success>Aco Update Complete</success>'));
		return true;
	}

/**
 * Check a node for existance, create it if it doesn't exist.
 *
 * @param string $path
 * @param string $alias
 * @param int $parentId
 * @return array Aco Node array
 */
	protected function _checkNode($path, $alias, $parentId = null) {
		$node = $this->Aco->node($path);
		if (!$node) {
			$this->Aco->create(array('parent_id' => $parentId, 'model' => null, 'alias' => $alias));
			$node = $this->Aco->save();
			$node['Aco']['id'] = $this->Aco->id;
			$this->out(__('Created Aco node: <success>%s</success>', $path), 1, Shell::VERBOSE);
		} else {
			$node = $node[0];
		}
		return $node;
	}

/**
 * Get a list of site setting categories.
 * @return array List of site setting categories.
 */
	public function getSiteSettingCategoryList() {
		return $this->SiteSettingCategory->find('list');
	}

/**
 * Check and Add/delete site setting categories.
 * @param  array  $root       The root node site_settings
 * @param  array  $categories List of stie setting categories
 */
	protected function _updateSiteSettingCategories($root, $categories) {
		foreach ($categories as $categoryId => $category) {
			$path = $this->rootNode . '/' . $category;
			$categoryNode = $this->_checkNode($path, $category, $root['Aco']['id']);
			$this->_checkSiteSettings($category, $categoryId, $categoryNode);
		}
		if ($this->_clean) {
			$categoryFlip = array_flip($categories);

			$this->Aco->id = $root['Aco']['id'];
			$categoryNodes = $this->Aco->children(null, true);

			foreach ($categoryNodes as $categoryNode) {
				$alias = $categoryNode['Aco']['alias'];

				if (!isset($categoryFlip[$alias])) {
					if ($this->Aco->delete($categoryNode['Aco']['id'])) {
						$this->out(__(
							'Deleted %s and all children',
							$this->rootNode . '/' . $categoryNode['Aco']['alias']
						), 1, Shell::VERBOSE);
					}
				}
			}
		}

		return;
	}

/**
 * Check and Add/delete site settings for a specific site setting category.
 *
 * @param string $controller
 * @param array $node
 * @param string $plugin Name of plugin
 * @return void
 */
	protected function _checkSiteSettings($category, $categoryId, $node) {
		$siteSettings = $this->getSiteSettingList($categoryId);
		if (empty($siteSettings)) {
			$this->err(__('Unable to get methods for "%s"', $category));
			return false;
		}

		foreach ($siteSettings as $siteSetting) {
			$path = $this->rootNode . '/' . $category . '/' . $siteSetting;
			$this->_checkNode($path, $siteSetting, $node['Aco']['id']);
		}

		if ($this->_clean) {
			$siteSettingNodes = $this->Aco->children($node['Aco']['id']);
			$siteSettingFlip = array_flip($siteSettings);

			foreach ($siteSettingNodes as $siteSetting) {
				if (!isset($siteSettingFlip[$siteSetting['Aco']['alias']])) {
					$this->Aco->id = $siteSetting['Aco']['id'];
					if ($this->Aco->delete()) {
						$path = $this->rootNode . '/' . $category . '/' . $siteSetting['Aco']['alias'];
						$this->out(__('Deleted Aco node: <warning>%s</warning>', $path), 1, Shell::VERBOSE);
					}
				}
			}
		}
		return true;
	}

/**
 * Get a list of all the site settings within a site setting category.
 * @param  int    $categoryId The id of the site setting category.
 * @return array              A list of site settings.
 */
	public function getSiteSettingList($categoryId) {
		return $this->SiteSetting->find(
			'list',
			[
				'conditions' => [
					'SiteSetting.site_setting_category_id' => $categoryId
				]
			]
		);
	}
}
