<?php

App::uses('AppHelper', 'View/Helper');

App::uses('InflectorExt', 'EvInflector.Lib');

class ProductsHelper extends AppHelper {

	public function __construct(View $View, $settings = array()) {
		$this->helpers[] = 'Number';

		$this->helpers['Form'] = array(
			'className' => 'EvForm.FormExt'
		);

		$this->helpers['Html'] = array(
			'className' => 'HtmlExt'
		);

		if (CakePlugin::loaded('EvInventory')) {
			$this->helpers['Inventory'] = array(
				'className' => 'EvInventory.Inventory'
			);
		}

		parent::__construct($View, $settings);
	}

/**
 * Checks the supplied values and returns a boolean values based on
 * whether the filter field should be marked as checked / selected
 *
 * @param string $productOptionGroupName
 * @param string $productOptionName
 * @return boolean
 */
	protected function _isFilterSelected(
		$productOptionGroupName = null,
		$productOptionName = null
	) {
		$selected = false;

		//Check if this group has any selected filters
		if (in_array($productOptionGroupName, $this->request->params['named'])) {
			//change name to url safe name
			$urlSafeProductOptionName = InflectorExt::slugUrlSafePreserveChars($productOptionGroupName . '-' . $productOptionName, true);
			$selected = array_key_exists(
				$urlSafeProductOptionName,
				$this->request->params['named']
			);
		}

		return $selected;
	}

/**
 * Contructs and returns unordered list element containing available
 * colour options ready to manipulate with CSS and Javascript
 *
 * @param array $optionGroup Contains the colour options to format
 * @param string $filterId The string to use as part of the filter checkboxes
 * @return string Contains the HTML ul element to d©2isplay on he DOM
 */
	protected function _buildColourFilterGroupFields(
		$optionGroup = [],
		$filterId = ''
	) {
		$filterOptionsHtml = '';

		if (! empty($optionGroup['FilterGroupOption'])) {
			foreach ($optionGroup['FilterGroupOption'] as $filterOptionItem) {
				$groupSelected = true;

				$optionItemHtml = $this->Html->tag('div', null, [
					'class' => 'truefilters hidden'
				]);

				foreach ($filterOptionItem['Option'] as $productOption) {
					$isChecked = $this->_isFilterSelected(
						$optionGroup['name'],
						$productOption['name']
					);

					$productOptionSafeName = InflectorExt::slugUrlSafePreserveChars(
						$productOption['name'],
						true
					);

					$checkboxId = $optionGroup['name'] . $productOptionSafeName . $filterId;

					if ($isChecked === false) {
						$groupSelected = false;
					}

					$optionItemHtml .= $this->Form->checkbox(
						'EvShopFilter.' . $optionGroup['FilterGroup']['name'] . '.' . $productOptionSafeName,
						[
							'id' => $checkboxId,
							'class' => 'required js-auto-submit',
							'value' => $productOption['name'],
							'checked' => $isChecked
						]
					);

					// Form->label() mangles the custom ID passed in
					$optionItemHtml .= $this->Html->tag('label', $productOption['name'], [
						'for' => $checkboxId
					]);
				}

				$optionItemHtml .= $this->Html->tag('/div');

				$optionItemHtml .= $this->Form->checkbox(false, [
					'class' => 'group-filter-check hidden',
					'id' => 'filterGf' . $filterOptionItem['id'],
					'checked' => $groupSelected,
					'value' => ''
				]);

				$optionItemHtml .= $this->Html->tag('label', null, [
					'for' => 'filterGf' . $filterOptionItem['id']
				]);

				$optionItemHtml .= $this->Html->tag(
					'div',
					$this->Html->tag('div', ''),
					[
						'class' => 'color-pick__item',
						'style' => 'background-color: #' . $filterOptionItem['data'] . ';'
					]
				);

				// wrap the $optionItemHtml markup in a li, ready for the DOM
				$filterOptionsHtml .= $this->Html->tag('li', $optionItemHtml);
			}
		}

		// return an ul element containing the contructions li elements
		return $this->Html->tag('ul', $filterOptionsHtml, ['class' => 'menu__section__content color-pick large']);
	}

/**
 * Contructs and return element containing filter group fields
 *
 * @param array $optionGroup Contains array of options to convert to HTML
 * @param string $filterId The unique string to use wihin checkbox elements
 * @return string Containing the filter group fields to show on the DOM
 */
	protected function _buildFilterGroupFields($optionGroup = [], $filterId = '') {
		$filterGroupFieldsHtml = '';

		foreach ($optionGroup['FilterGroupOption'] as $filterOptionItem) {
			$groupSelected = true;

			$productOptionsHtml = '';

			foreach ($filterOptionItem['Option'] as $productOption) {
				$isChecked = $this->_isFilterSelected(
					$optionGroup['name'],
					$productOption['name']
				);

				$productOptionSafeName = InflectorExt::slugUrlSafePreserveChars(
					$productOption['name'],
					true
				);

				$checkboxId = $optionGroup['name'] . $productOptionSafeName . $filterId;

				if ($isChecked === false) {
					$groupSelected = false;
				}

				$productOptionsHtml .= $this->Form->checkbox(
					'EvShopFilter.' . $optionGroup['FilterGroup']['name'] . '.' . $productOptionSafeName,
					[
						'id' => $checkboxId,
						'class' => 'required js-auto-submit',
						'value' => $productOption['name'],
						'checked' => $isChecked
					]
				);

				// Form->label() mangles the custom ID passed in
				$productOptionsHtml .= $this->Html->tag(
					'label',
					$productOption['name'],
					['for' => $checkboxId]
				);
			}

			$filterGroupHtml = $this->Html->tag('div', $productOptionsHtml, [
				'class' => 'truefilters hidden'
			]);

			$filterGroupHtml .= $this->Form->checkbox(false, [
				'class' => 'group-filter-check',
				'id' => 'filterGf' . $filterOptionItem['id'],
				'checked' => $groupSelected,
				'value' => ''
			]);

			$filterGroupHtml .= $this->Html->tag('label', $filterOptionItem['name'], [
				'for' => 'filterGf' . $filterOptionItem['id']
			]);

			$filterGroupFieldsHtml .= $this->Html->tag('div', $filterGroupHtml, [
				'class' => 'checkbox'
			]);
		}

		return $filterGroupFieldsHtml;
	}

/**
 * Contructs and returns form checkbox elements to use within the filter fields DOM
 *
 * @param array $optionGroup Contains array of options to convert to HTML
 * @param string $filterId The unique string to use wihin checkbox elements
 * @return string Containing the filter group fields to show on the DOM
 */
	protected function _buildFilterFields($optionGroup = [], $filterId = '') {
		$filterFieldsHtml = '';

		foreach ($optionGroup['Option'] as $option) {
			$isChecked = $this->_isFilterSelected(
				$optionGroup['name'],
				$option['name']
			);

			$optionSafeName = InflectorExt::slugUrlSafePreserveChars($option['name'],
				true
			);

			$checkboxId = $optionGroup['name'] . $optionSafeName . $filterId;

			$checkbox = $this->Form->checkbox(
				'EvShopFilter.' . $optionGroup['name'] . '.' . $optionSafeName,
				[
					'id' => $checkboxId,
					'class' => 'js-auto-submit',
					'value' => $option['name'],
					'checked' => $isChecked
				]
			);

			// Form->label() mangles the custom ID passed in
			$label = $this->Html->tag('label', $option['name'], [
				'for' => $checkboxId
			]);

			$filterFieldsHtml .= $this->Html->tag('div', $checkbox . $label, [
				'class' => 'checkbox',

			]);
		}

		return $this->Html->tag('div', $filterFieldsHtml, [
			'class' => 'menu__section__content collapse',
			'id' => 'menu-collapse-content-sub-' . preg_replace('/[^\w-]+/', '', $optionGroup['name']),
			'aria-expanded' => 'false'
		]);
	}

/**
 * Loops around supplied array of price options and updated the value of each
 * item to show a currency representation of the price
 *
 * @param array $priceOptions An array of price options to format
 * @return array Contains price options with formatted array values
 */
	public function formatPriceOptions($priceOptions = []) {
		if (! empty($priceOptions)) {
			foreach ($priceOptions as $key => $option) {
				$priceOptions[$key] = $this->Number->currency($option, null, ['places' => 0]);
			}
		}
		return $priceOptions;
	}

/**
 * for a product with single variant, extract the requests data
 *
 * @param   string      The column to extract
 * @param   array       The Data array
 * @return  mixed       The extracted data or bool false if multiple variants
 */
	public function singleVariantData($item, $data) {
		if (count($data['Variant']) > 1) {
			return false;
		}

		return Hash::get($data, 'Variant.0.' . $item);
	}

/**
 * get the option groups for the variants
 *
 * @param   array   Product Data array
 * @return  array   Array of option groups
 */
	public function getOptionGroups($data) {
		if (! empty($data['Variant']['0']['Option'])) {
			return Hash::combine($data['Variant'], '{n}.Option.{n}.OptionGroup.id', '{n}.Option.{n}.OptionGroup.name');
		}

		return array();
	}

/**
 * gets all available option types and the available options
 * Returns an array in the format
 * [
 *      optionGroupName => [
 *          'label' => 'Label',
 *          // This below is ideal for passing directly to a select inputs options parameter
 *          'options' => [
 *              'optionId' => 'Option Name'
 *          ]
 *          // The additional data field assiciated with an option.
 *          // Given in a separate lookup array so the above can be used to populate selects
 *          'optionData' => [
 *              'optionId' => 'Other Data'
 *          ]
 *
 *      ]
 * ]
 */
	public function getAllAvailableOptions($data) {
		$variantOptions = Hash::extract($data, 'Variant.{n}.Option.{n}');
		$variantOptions = Hash::sort($variantOptions, '{n}.sequence', 'asc');
		$options = [];

		foreach ($variantOptions as $option) {
			if (!isset($options[$option['OptionGroup']['id']])) {
				$options[$option['OptionGroup']['id']] = $this->_getDataForAvailableOptionGroup($option['OptionGroup']);
			}

			if (!isset($options[$option['OptionGroup']['id']]['options'][$option['id']])) {
				$options[$option['OptionGroup']['id']]['options'][$option['id']] = $this->_getDataForAvailableOption($option);
			}
		}

		uasort($options, function ($a, $b) {
			if ($a['sequence'] > $b['sequence']) {
				return 1;
			} elseif ($a['sequence'] < $b['sequence']) {
				return -1;
			}

			return 0;
		});

		return $options;
	}

/**
 * Show an option field.
 *
 * This rerurns the element that handles the field type for anoption field. You
 * can optionally override attributes on the field or label, and these are passed
 * you your custom option group field element.
 *
 * @param int $id Id of the option group
 * @param array  $option The array of option data
 * @param mixed $customAttributes An optional array of custom field attributes
 * @param mixed $labelAttributes An optional array of custom label attributes
 * @return View Returns the field type element being used by the option group
 */
	public function showOptionField($id, $option, $customAttributes = false, $labelAttributes = false) {
		if (! empty($option['field_type'])) {

			$fieldOptions = [];
			// Fields like radios need a simple value > label array so we'll
			// create one from the individual options.
			if (! empty($option['options'])) {
				foreach ($option['options'] as $item) {
					if (! empty($item['name']) && ! empty($item['value'])) {
						$fieldOptions[$item['value']] = $item['name'];
					}
				}
			}

			return $this->element(
				'EvShop.OptionGroupFields/' . $option['field_type'],
				[
					'field' => $option,
					'fieldName' => 'Product.OptionGroup.' . $id . '',
					'attrs' => $customAttributes,
					'labelAttrs' => $labelAttributes,
					'fieldOptions' => $fieldOptions
				]
			);
		}
	}

/**
 * Get the data for an available option group.
 *
 * @param array $optionGroup The data of the current available option group.
 * @return array The data to be used
 */
	protected function _getDataForAvailableOptionGroup($optionGroup) {
		$optionGroupData = [
			'label' => $optionGroup['name'],
			'sequence' => $optionGroup['sequence'],
			'field_type' => isset($optionGroup['field_type']) ? $optionGroup['field_type'] : null,
			'options' => []
		];

		return $optionGroupData;
	}

/**
 * Get the data for an available option.
 *
 * @param array $optionValue The data of the current available option.
 * @return array The data to be used
 */
	protected function _getDataForAvailableOption($optionValue) {
		$optionData = [
			'name' => $optionValue['name'],
			'data-value' => $optionValue['data'],
			'data-sequence' => $optionValue['sequence'],
			'value' => $optionValue['id'],
		];

		return $optionData;
	}

/**
 * Initialises the JS required for the option paths funtionality.
 */
	public function initOptionPaths($data) {
		$this->Html->script( 'EvShop.optionFilter.js', ['inline' => false, 'once' => true] );
		$this->Html->scriptStart( ['inline' => false ] );
		?>

		initShopOptions({
			"availablePaths": <?= json_encode($this->getAvailableOptionPaths($data)) ?>
		});

		<?php
		$this->Html->scriptEnd();
	}

/**
 * Returns an array containing all possible selection paths for the available options to get a variant.
 *
 * Return array is in the format
 * [
 *      'options' => [
 *          // All possible options that could be selected next
 *          'optionGroupId' => [
 *              'optionId' => [
 *                  // This will either be another nested instance of this structure or if all options have been selected, it will be info about a specific variant
 *                  // Check for a variant_id to detect. If it's a variant...
 *                  'variant_id' => The variant id,
 *                  'price' => The result of getVariantPrice for this variant
 *                  'wasPriceFormatted' => A formatted version of the wasPrice which is usually the regular price but can be RRP if no regular price and on sale. Null if neither.
 *                  'nowPriceFormatted' => The current price formatted as the current currency
 *                  'availability' => Either 'good', 'low', 'delayed' or 'none' (see getVariantAvailability)
 *              ]
 *          ]
 *      ],
 *      // The below are data based on all possible combinations of options after the user's current selection
 *      'price' => The price array with the lowest price from all possible options
 *      'minPriceFormatted' => '£20.00' // The formatted version using the current currency. Can be use for "Now £XX" or "From £XX" pricing
 *      'minPriceWasFormatted' => '£50.00' // The formatted version using the current currency. Can be use for "Was £XX" pricing
 *      'availability' => 'good' // Either 'good', 'some_low', 'low' or 'none' depending on stock and warning settings (see _getAvailabilityFromOptionsPath)
 * ]
 *
 * @param array $data The data of a product.
 * @return array The option paths a user can take to get a variant.
 */
	public function getAvailableOptionPaths($data) {
		$Product = EvClassRegistry::init('EvShop.Product');
		$cachedPaths = $Product->fetchFromCache($data['Product']['id'], 'optionPaths_' . CakeSession::read('EvCurrency.currencyCode'));

		if (!empty($cachedPaths)) {
			//If cached paths have been found then return them
			return $cachedPaths;
		}

		$paths = [];

		if (!empty($data['Variant'])) {
			foreach ($data['Variant'] as $variantIndex => $variant) {
				if (!empty($variant['Option'])) {
					$variantPaths = $this->_getAvailableOptionPaths($data, $variant, $variant['Option']);

					foreach ($variantPaths as $key => $variantPath) {
						if (!empty($paths[$key])) {
							$paths[$key] = $this->_mergePathElement($paths[$key], $variantPath);
						} else {
							$paths[$key] = $variantPath;
						}
					}
				}
			}
		}

		if (!empty($paths)) {
			//The paths are returned as . separated keys. Expand them to get a full array.
			$paths = Hash::expand($paths);

			//Add root data.
			$minPrice = $this->_getMinPriceFromOptionsPath($paths);
			$paths = $this->_addAdditionalVariantDataToPathRoot(
				Hash::merge($paths, [
					'price' => $minPrice,
					'minPriceFormatted' => $this->Number->currency( (float)$minPrice['lowest_price'] ),
					'minPriceWasFormatted' => !empty((float)$minPrice['higher_price']) ? $this->Number->currency((float)$minPrice['higher_price']) : null,
					'availability' => $this->_getAvailabilityFromOptionsPath($paths),
				]),
				$data
			);

			$Product->cache($data['Product']['id'], 'optionPaths_' . CakeSession::read('EvCurrency.currencyCode'), $paths);
		}

		return $paths;
	}

/**
 * Merge an intermediate element
 *
 * @param array $original The original path data
 * @param array $new Path data to merge in
 * @return array The merged element
 */
	protected function _mergePathElement($original, $new) {
		$availabilityStates = [
			'none' => 0,
			'low' => 1,
			'some_low' => 2,
			'good' => 3,
		];

		if ($availabilityStates[$new['availability']] > $availabilityStates[$original['availability']]) {
			$original['availability'] = $new['availability'];
		}

		if ($new['price']['lowest_price'] < $original['price']['lowest_price']) {
			$original['price'] = $new['price'];
			$original['minPriceFormatted'] = $this->Number->currency((float)$original['price']['lowest_price']);
			$original['minPriceWasFormatted'] = !empty((float)$original['price']['higher_price']) ? $this->Number->currency((float)$original['price']['higher_price']) : null;
		}

		return $original;
	}

/**
 * Recursively build an array of possible option paths to get a variant.
 *
 * @param array  $data    Product data.
 * @param array  $variant Contained variant data of a product.
 * @param array  $options Contained option data of a variant.
 * @param string $path    The current dot separated path to the current option.
 * @return array The available option paths.
 */
	protected function _getAvailableOptionPaths($data, $variant, $options, $path = '') {
		$paths = [];

		foreach ($options as $optionIndex => $option) {
			$optionId = $option['id'];
			$optionGroupId = $option['OptionGroup']['id'];

			//Remove the current option from the current branch of available options.
			$recurseOptions = $options;
			unset($recurseOptions[$optionIndex]);

			//Build the path of the current element.
			$newPath = 'options.' . $optionGroupId . '.' . $optionId;
			if (!empty($path)) {
				$newPath = $path . '.' . $newPath;
			}

			if (!empty($recurseOptions)) {
				//There are still branches left to explore so recurse again to get the child branches.
				$paths = $paths + $this->_getAvailableOptionPaths($data, $variant, $recurseOptions, $newPath);

				//Add data to the current path element. Uses older methods to get minimum price in the tree
				$optionsBranch = Hash::get(Hash::expand($paths), $newPath);

				$minPrice = $this->_getMinPriceFromOptionsPath($optionsBranch);
				$paths[$newPath] = $this->_addAdditionalVariantDataToPathElement(
					[
						'price' => $minPrice,
						'minPriceFormatted' => $this->Number->currency((float)$minPrice['lowest_price']),
						'minPriceWasFormatted' => !empty((float)$minPrice['higher_price']) ? $this->Number->currency((float)$minPrice['higher_price']) : null,
						'rrpFormatted' => $this->Number->currency((float)$minPrice['rrp']),
						'availability' => $this->_getAvailabilityFromOptionsPath($optionsBranch),
						// Temporarily add the options array in to make it backwards compatible with previous versions
						'options' => $optionsBranch['options'],
					],
					$data
				);
				unset($paths[$newPath]['options']);
			} else {
				//We have no more options to branch on so this must be a variant selection.
				$pricing = $this->getVariantPrice($variant['VariantPricing']);
				$variantInfo = $this->_addAdditionalVariantDataToPathEnd(
					[
						'variant_id' => $variant['id'],
						'price' => $pricing,
						'wasPriceFormatted' => !empty((float)$pricing['higher_price']) ? $this->Number->currency((float)$pricing['higher_price']) : null,
						'nowPriceFormatted' => $this->Number->currency($pricing['lowest_price']),
						'rrpFormatted' => $this->Number->currency((float)$pricing['rrp']),
						'availability' => !empty($variant['Inventory']) ? $this->getVariantAvailability($variant) : 'good',
					],
					$data,
					$variant
				);

				$paths[$newPath] = $variantInfo;
			}
		}

		return $paths;
	}

/**
 * Override this to add any additional data to the path endpoints. This allows inserting extra data without trying to override the whole function.
 * @param [type] $endElement The variant description. This will be publicly viewable so don't include anything private like keys etc.
 * @param [type] $data       The data the array is being built from
 * @param [type] $data       The current variant being described
 */
	protected function _addAdditionalVariantDataToPathEnd($endElement, $data, $variant) {
		return $endElement;
	}

/**
 * Override this to add any additional data to the path elements that are not actual variants. This allows inserting extra data without trying to override the whole function.
 * @param [type] $pathElement The current path point including all children in the 'options' field. This will be publicly viewable so don't include anything private like keys etc.
 * @param [type] $data       The data the array is being built from
 */
	protected function _addAdditionalVariantDataToPathElement($pathElement, $data) {
		return $pathElement;
	}

/**
 * Override this to add any additional data to the root part element. This allows inserting extra data without trying to override the whole function.
 * @param [type] $pathElement The current path point including all children in the 'options' field. This will be publicly viewable so don't include anything private like keys etc.
 * @param [type] $data       The data the array is being built from
 */
	protected function _addAdditionalVariantDataToPathRoot($pathElement, $data) {
		return $pathElement;
	}

/**
 * Finds the minimum price when given the starting point on an available paths tree.
 */
	protected function _getMinPriceFromOptionsPath($optionPaths) {
		$minPrice = null;

		foreach ($optionPaths['options'] as $optionType => $optionValues) {
			foreach ($optionValues as $optionValue => $optionValueData) {
				if ($minPrice === null || (float)$optionValueData['price']['lowest_price'] < (float)$minPrice['lowest_price']) {
					$minPrice = $optionValueData['price'];
				}
			}
		}

		return $minPrice;
	}

/**
 * Finds an average availability based on current options
 *
 * Returns either:
 *  good - All variants along the given path are showing plenty of stock
 *  some_low - Some of the variants are showing low or no stock. There is no some_none as these two messages will likely conflict
 *  low - All variants are showing low stock
 *  none - All variants along this path are out of stock
 */
	protected function _getAvailabilityFromOptionsPath($optionPaths) {
		$currentLevelCount = [
			'good' => 0,
			'some_low' => 0,
			'low' => 0,
			'none' => 0
		];

		foreach ($optionPaths['options'] as $optionType => $optionValues) {
			foreach ($optionValues as $optionValue => $optionValueData) {
				if (!empty($optionValueData['availability'])) {
					switch($optionValueData['availability']) {
						case 'delayed':
							// Show delayed as good for variant groups
							$currentLevelCount['good']++;
							break;
						default:
							$currentLevelCount[$optionValueData['availability']]++;
							break;
					}
				}
			}
		}

		if ($currentLevelCount['good'] == 0 && $currentLevelCount['some_low'] == 0 && $currentLevelCount['low'] == 0) {
			// All variants along this path are out of stock
			return 'none';
		} elseif ($currentLevelCount['good'] == 0 && $currentLevelCount['some_low'] == 0) {
			// All variants are low on stock
			return 'low';
		} elseif ($currentLevelCount['low'] > 0 || $currentLevelCount['some_low'] > 0 || $currentLevelCount['none'] > 0) {
			// All variants are low or out of stock
			return 'some_low';
		}

		return 'good';
	}

/**
 * given the variant array and the option group
 * return the option name
 *
 * @param   array   Variant Array
 * @param   int     Option group Id
 * @return  string  Option value
 */
	public function getOption($Variant, $groupId) {
		return Hash::get($Variant, 'Option.' . $groupId . '.name', '-');
	}

/**
 * Given an array of VariantPricings, the pricing that matches the current currency ID will be found
 * and an array will be returned with the price. If the variant pricing contains a sale_price then
 * that will be returned as well as the lowest_price and the original price returned as the price.
 * @param  array $pricings An array of variant pricings for a specific variant
 * @return array           An array containing the price and the sale_price if it exists.
 */
	public function getVariantPrice($pricings) {
		$possiblePricings = Hash::combine($pricings, '{n}.currency_id', '{n}');
		$currencyId = CakeSession::read('EvCurrency.currencyId');

		$pricing = $possiblePricings[$currencyId];

		$originalPrice = $pricing['price'];
		$rrp = $pricing['rrp'];
		if (Configure::read('EvShop.displayTaxAsVat')) {
			if (!empty($pricing['price_incTax'])) {
				$originalPrice = $pricing['price_incTax'];
			}

			if (!empty($pricing['rrp_incTax'])) {
				$rrp = $pricing['rrp_incTax'];
			}
		}

		if (isset($pricing['sale_price']) && !empty((float)$pricing['sale_price'])) {
			// Logic flip for legacy sites using RRP as "Was" price when items are on sale
			if (Configure::read('EvShop.preferRrpForWasPrice') === true) {
				$higherPrice = (!empty($rrp) ? $rrp : (!empty($originalPrice) ? $originalPrice : null));
			} else {
				$higherPrice = (!empty($originalPrice) ? $originalPrice : (!empty($rrp) ? $rrp : null));
			}

			return [
				'price' => $originalPrice,
				'sale_price' => $pricing['lowest_price'],
				'rrp' => $rrp,
				'lowest_price' => $pricing['lowest_price'],
				'higher_price' => $higherPrice
			];
		} else {
			return [
				'price' => $pricing['lowest_price'],
				'rrp' => $rrp,
				'lowest_price' => $pricing['lowest_price'],
				'higher_price' => (!empty($rrp) ? $rrp : null),
			];
		}
	}

/**
 * Will take into account the warning and oos actions to return either 'none', 'delayed', 'low' or 'good'
 */
	public function getVariantAvailability($inventory) {
		if (!$this->Inventory->allowPurchase($inventory)) {
			return 'none';
		} else {
			if ($this->Inventory->showDelayedMessage($inventory)) {
				return 'delayed';
			} elseif ($this->Inventory->showLowStockMessage($inventory)) {
				return 'low';
			}
		}

		// At this point the stock may be low or gone completely but we won't tell the customer
		// because the warning actions are not set to do so
		return 'good';
	}

/**
 * Given a product listing, show either the image which is a closest match to the filters or if this is unset
 * return the listing image
 * @param  array $productListing A product listing item array
 * @return array                 An image array or null if no suitable image found
 */
	public function getListingImage($productListing) {
		if (!empty($productListing['Image']['id'])) {
			// This will be the closest match to the filters as selected by the query
			return $productListing['Image'];
		} elseif (!empty($productListing['ListingImage'][0])) {
			// Fall back to the regular listing image if no suitable variant image.
			return $productListing['ListingImage'][0];
		} else {
			// No images available
			return null;
		}
	}

/**
 * Given a set of variant images, remove the restricted images so that only the generic images are left.
 * @param  array $allVariantImages An array of variant images
 * @return array                  An array without an restricted images in
 */
	public function getGenericVariantImages($allVariantImages) {
		foreach ($allVariantImages as $imageKey => $image) {
			if (isset($image['is_restricted']) && $image['is_restricted']) {
				unset($allVariantImages[$imageKey]);
			}
		}

		return $allVariantImages;
	}

/**
 * Given a set of variant images, remove the generic images so that only the restricted images are left.
 * Also remove any restricted images that don't have any options associated with them.
 * @param  array $allVariantImages An array of variant image
 * @return array                   An array without any generic images in,
 */
	public function getRestrictedVariantImages($allVariantImages) {
		foreach ($allVariantImages as $imageKey => $image) {
			if (!isset($image['is_restricted']) || !$image['is_restricted']) {
				unset($allVariantImages[$imageKey]);
			}

			if (!isset($image['VariantImageOption']) || empty($image['VariantImageOption'])) {
				unset($allVariantImages[$imageKey]);
			}
		}

		return $allVariantImages;
	}

/**
 * Given a product variant and a set of variant images, find all the images that are applicable to be displayed
 * for the variant. The variant images need to have options associated with them otherwise they will not be
 * returned.
 * @param  array $variant       An array containing the information about a specific variant
 * @param  array $variantImages An array of variant images.
 * @return array                An array of variant images that are applicable to the specified variant.
 */
	public function getRestrictedImagesForVariant($variant, $variantImages) {
		$imageArray = [];

		if (isset($variant['Option']) && !empty($variant['Option'])) {
			$allVariantOptions = Hash::extract($variant['Option'], '{n}.id');

			$restrictedImages = $this->getRestrictedVariantImages($variantImages);

			foreach ($restrictedImages as $image) {
				$currentGroupsChecked = [];

				foreach ($image['VariantImageOption'] as $imageOption) {
					if (!isset($currentGroupsChecked[$imageOption['option_group_id']])) {
						$currentGroupsChecked[$imageOption['option_group_id']] = false;
					}

					if (in_array($imageOption['option_id'], $allVariantOptions)) {
						$currentGroupsChecked[$imageOption['option_group_id']] = true;
					}
				}

				if (!in_array(false, $currentGroupsChecked)) {
					$imageArray[] = $image;
				}
			}
		}

		return $imageArray;
	}

/**
 * Gets the most relevant image based on variant image restrictions. Usually used in the basket or when showing orders.
 *
 * @param Variant $variant The variant to match.
 * @param array $variantImagePath The path within the variant array where the restricted images can be found
 * @param array $listingImagePath The path within the variant array where the product listing image can be found
 * @param array $fallbackImage The very last resort image. Usually a generic image.
 * @return mixed The image array to use. If no suitable image is found the fallback will be returned.
 */
	public function getMostRelevantImageForVariant(
		$variant,
		$variantImagePath = 'Product.GalleryVariantImage',
		$listingImagePath = 'Product.ListingImage',
		$fallbackImage = []
	) {
		$restrictedImages = Hash::extract($variant, $variantImagePath);
		if (!empty($restrictedImages)) {
			// Filter the restricted images to ones applicable to this variant
			$restrictedImages = $this->getRestrictedImagesForVariant($variant, $restrictedImages);
		}

		if (!empty($restrictedImages)) {
			// All images in this list are relevant so just take the first
			$returnImage = $restrictedImages[0];
		} else {
			$listingImages = Hash::extract($variant, $listingImagePath);
			if (!empty($listingImages[0])) {
				// Should only be one listing image
				$returnImage = $listingImages[0];
			} else {
				$returnImage = $fallbackImage;
			}
		}

		return $returnImage;
	}

/**
 * Basic GETTER for getting sale price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @param $incVat Defaults to false, if set to true INC vat price is returned
 * @return formatted currency Sale price
 */
	public function getSalePrice($product, $incVat = false) {
		$currencies = $this->_View->viewVars['currencies'];
		if (!empty($product['was_price_ex_vat'])) {
			if ($incVat && !empty($product['was_price'])) {
				return $this->Number->currency($product['was_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
			} else {
				return $this->Number->currency($product['was_price_ex_vat'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
			}
		} else {
			return false;
		}
	}

/**
 * Basic GETTER for getting RRP price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @param $incVat Defaults to false, if set to true INC vat price is returned
 * @return  formatted currency RRP price
 */
	public function getRrpPrice($product, $incVat = false) {
		$currencies = $this->_View->viewVars['currencies'];
		if ($incVat) {
			return $this->Number->currency($product['rrp_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
		} else {
			return $this->Number->currency($product['rrp_price_ex_vat'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
		}
	}

/**
 * Basic GETTER for getting price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @return  formatted currency price
 */
	public function getPrice($product) {
		return $this->getProductPrice($product);
	}

/**
 * Basic GETTER for getting price of the product including VAT or not.
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product variant pricing array
 * @return  formatted currency price
 */
	public function getProductPrice($product, $incVat = false) {
		$currencies = $this->_View->viewVars['currencies'];

		if ($incVat && !empty($product['price_incTax'])) {
			return $this->Number->currency($product['price_incTax'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
		}

		if (!empty($product['price'])) {
			return $this->Number->currency($product['price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
		}

		//If the price isn't available on the product then check the variants for the lowest price.
		if ($incVat) {
			$productPrices = Hash::extract(
				$product,
				'Variant.{n}.VariantPricing.{n}[currency_id=' . CakeSession::read('EvCurrency.currencyId') . '].lowest_price'
			);

			if (!empty($productPrices)) {
				sort($productPrices);
				$lowestPrice = array_shift($productPrices);
				return $this->Number->currency($lowestPrice, $currencies[CakeSession::read('EvCurrency.currencyId')]);
			}
		}

		return null;
	}

/**
 * Basic GETTER for getting max price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @return  formatted currency price
 */
	public function getMaxPrice($product) {
		$currencies = $this->_View->viewVars['currencies'];
		return $this->Number->currency($product['max_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
	}

/**
 * Returns true if there is a WAS price to the product
 * @param  Product  $product Product array
 * @return boolean          Returns true if the item is on sale
 */
	public function isSalePrice($product) {
		if (empty($product['was_price_ex_vat'])) {
			return false;
		}

		return ($product['was_price_ex_vat'] > 0);
	}

/**
 * Constructs and returns the html for the products filter form, usually used as
 * part of the sidebar displayed on prouct index templates
 *
 * @param array $filterOptions Contains the filter options to covert to filter elements
 * @param array $priceOptions Contains the price options to covert to price elements
 * @param string $filterTitle The title of the filter element
 * @param string $filterId The id string to use for the collapse functionality
 * @param array $clearLink The array to use when producing the clear link, contains name and url
 * @return string Contains the HTML to display on the template
 */
	public function buildFilterForm(
		$filterOptions = [],
		$priceOptions = [],
		$filterTitle = 'Filter Products',
		$filterId = 'Filter',
		$clearLink = []
	) {
		$filterHtml = $this->Html->tag('div', null, ['class' => 'menu']);

		if (!empty($filterTitle)) {
			$filterTitleLink = $this->Html->link(
				$filterTitle,
				'#menu-collapse-content-' . $filterId,
				[
					'class' => 'menu__heading',
					'data-toggle' => 'collapse'
				]
			);

			$filterHtml .= $this->Html->tag('div', $filterTitleLink, [
				'class' => 'menu__header'
			]);
		}

		// Collapse block open
		$filterHtml .= $this->Html->tag('div', null, [
			'aria-expanded' => false,
			'class' => 'menu__content collapse',
			'id' => 'menu-collapse-content-' . $filterId
		]);

		$filterHtml .= $this->Form->create('EvShopFilter', [
			'inputDefaults' => [
				'div' => 'form-group',
				'label' => [
					'class' => 'control-label'
				],
				'wrapInput' => '',
				'class' => 'form-control'
			],
			'id' => 'Filter' . $filterId,
			'class' => ''
		]);

		if (! empty($filterOptions)) {
			foreach ($filterOptions as $optionGroup) {
				if (!empty($optionGroup['Option'])) {
					// Group container open
					$filterHtml .= $this->Html->tag('div', null, ['class' => 'menu__section menu__section--filter-' . InflectorExt::slug(strtolower($optionGroup['name']))]);
					// Menu heading open
					$filterHtml .= $this->Html->tag('div', null, [
						'class' => 'menu__sub-heading collapsed',
						'data-toggle' => 'collapse',
						'data-target' => '#menu-collapse-content-sub-' . preg_replace('/[^\w-]+/', '', $optionGroup['name'])
					]);

					$filterHtml .= $optionGroup['name'];

					if (! empty($optionGroup['tooltip'])) {
						$filterHtml .= ' ' . $this->Html->tag('i', '', [
							'class' => 'fa fa-info-circle',
							'data-toggle' => 'tooltip',
							'data-placement' => 'right',
							'title' => $optionGroup['tooltip']
						]);
					}

					// Menu heading close
					$filterHtml .= $this->Html->tag('/div');

					if (isset($optionGroup['FilterGroup'])) {
						if (
							! empty($optionGroup['option_group_id']) &&
							Configure::check('EvShop.colorOptionGroupId') &&
							$optionGroup['option_group_id'] == Configure::read('EvShop.colorOptionGroupId')
						) {
							$filterHtml .= $this->_buildColourFilterGroupFields(
								$optionGroup,
								$filterId
							);
						} else {
							$filterHtml .= $this->_buildFilterGroupFields(
								$optionGroup,
								$filterId
							);
						}
					} else {
						$filterHtml .= $this->_buildFilterFields(
							$optionGroup,
							$filterId
						);
					}

					// Group container close
					$filterHtml .= $this->Html->tag('/div');
				}
			}
		}

		// Add in the price filters, if populated
		if (! empty($priceOptions)) {
			// Group container open
			$filterHtml .= $this->Html->tag('div', null, ['class' => 'menu__section menu__section--filter-price']);

			// Menu heading open
			$filterHtml .= $this->Html->tag('div', __('Price'), [
				'class' => 'menu__sub-heading'
			]);

			//option body open
			$filterHtml .= $this->Html->tag('div', null, ['class' => 'menu__section__content price-select']);
			$priceOptionsFormatted = $this->formatPriceOptions( $priceOptions );
			$filterHtml .= $this->Form->select(
				'EvShopFilter.min_price',
				$priceOptionsFormatted,
				[
					'class' => 'required js-auto-submit',
					'empty' => 'Any',
					'escape' => false
				]
			);
			$filterHtml .= $this->Html->tag('span', 'to');
			$filterHtml .= $this->Form->select(
				'EvShopFilter.max_price',
				$priceOptionsFormatted,
				[
					'class' => 'required js-auto-submit',
					'empty' => 'Any',
					'escape' => false
				]
			);

			// Option body close
			$filterHtml .= $this->Html->tag('/div');

			// Group container close
			$filterHtml .= $this->Html->tag('/div');
		}

		// Add the clear options button

		if (!empty($clearLink)) {
			if (empty($clearLink['name'])) {
				$clearLink['name'] = 'Clear All Filters';
			}

			$clearFilterLink = $this->Html->link(
				$clearLink['name'],
				$clearLink['url'],
				[
					'class' => 'link--clear',
					'escape' => false
				]
			);

			$filterHtml .= $this->Html->tag('div', $clearFilterLink, [
				'class' => 'menu__section menu__section--clear-filters'
			]);
		}

		$filterHtml .= $this->Form->end();

		// Collapse block close
		$filterHtml .= $this->Html->tag('/div');

		// Main container close
		$filterHtml .= $this->Html->tag('/div');

		return $filterHtml;
	}

/**
 * Get an attribute from a product array
 *
 * @param array $product The product to search. Must have contained ProductAttribute and ProductAttributeGroup
 * @param int|string $attributeGroupId Either the attribute group id or system name
 * @return array The attribute
 */
	public function getAttribute($product, $attributeGroupId) {
		if (is_numeric($attributeGroupId)) {
			$compareField = 'id';
		} else {
			$compareField = 'system_name';
		}

		if (!empty($product['ProductAttribute'])) {
			foreach ($product['ProductAttribute'] as $attribute) {
				if ($attribute['ProductAttributeGroup'][$compareField] === $attributeGroupId) {
					return $attribute;
				}
			}
		}

		return null;
	}

/**
 * Get an array of variant options. The options are returned in the format Option Group Id -> Option Id and
 * are in the order of the option group sequence.
 *
 * @param array $product The product to search. Must have contained Variant > Option > Option Group
 * @param int $variantId The variant Id
 * @return array A key value pair of optiongroupid => optionid
 */
	public function getVariantOptions($product, $variantId) {
		if (empty($variantId)) {
			return [];
		}

		$variant = Hash::extract($product, "Variant.{n}[id=$variantId]");
		$variant = array_shift($variant);

		if (empty($variant['Option'])) {
			return [];
		}

		$variantOptions = $variant['Option'];
		$variantOptions = Hash::sort($variantOptions, '{n}.OptionGroup.sequence');

		return Hash::combine($variantOptions, '{n}.option_group_id', '{n}.id');
	}

/**
 * Get cached element or create a new one and cache it
 *
 * @param int $productId The product ID to cache against
 * @param string $elementName The element name
 * @param string $elementData The data you'd normally pass to the element
 * @param array $identifiers These will be merged to create a unique key for the cached element. You do not need to include the element name or product ID but can include things like a currencyID.
 * @return string The rendered element.
 */
	public function cachedElement($productId, $elementName, $elementData, $identifiers = []) {
		// Sort the identifiers incase the same values were passed in a different order
		ksort($identifiers);

		// Make a cache key from the element name and identifier values
		$cacheKey = $elementName;
		if (!empty($identifiers)) {
			$cacheKey .= '_' . implode('_', $identifiers);
		}

		// Check for an existing cached element
		$Product = EvClassRegistry::init('EvShop.Product');
		$element = $Product->fetchFromCache($productId, $cacheKey);

		if (empty($element)) {
			// Not found - build the element and cache it now.
			$element = $this->_View->element($elementName, $elementData);
			$Product->cache($productId, $cacheKey, $element);
		}

		return $element;
	}

}
