<?php
App::uses('EvShopAppModel', 'EvShop.Model');
App::uses('CakeNumber', 'Utility');
/**
 * Product Model
 *
 * @property Brand $Brand
 * @property Variant $Variant
 * @property Category $Category
 */
class Product extends EvShopAppModel {

	public $actsAs = array(
		'Routable.Routable' => array(
			'config' => 'EvShop',
			'alias' => 'product/:displayField'
		),
		'MetaData.Meta',
		'EvTemplates.Template' => array(
			'formInject' => true,
			'restrictTo' => 'Product'
		),
		'EvShop.ProductCopyable' => [
			'recurse_level' => 2,
		],
	);

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * A cached copy of the tax levels to avoid hitting the DB too many times
	 * @var array
	 */
	protected $_taxLevels = null;

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'A product name must be entered'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * belongsTo associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Brand' => array(
			'className' => 'EvShop.Brand'
		),
	);

	/**
	 * hasMany associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'Variant' => array(
			'className' => 'EvShop.Variant',
			'cascade' => true,
			'dependent' => true
		),
		'CategoriesProduct' => array(
			'className' => 'EvShop.CategoriesProduct',
			'order' => 'CategoriesProduct.sequence ASC'
		),
	);

	public $hasAndBelongsToMany = [
		'ProductAttribute' => [
			'className' => 'EvShop.ProductAttribute',
			'joinTable' => 'ev_shop_product_attributes_products'
		]
	];

	/**
	 * image slots
	 */
	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => false
		)
	);

	/**
	 * redefine constructor to check for EvTax
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (CakePlugin::loaded('EvTax')) {
			$this->actsAs['EvTax.Taxable'] = array(
				'formInject' => true
			);

			$this->belongsTo['TaxLevel'] = array(
				'className' => 'EvTax.TaxLevel'
			);
		}

		parent::__construct($id, $table, $ds);
	}

/**
 * Checks if the Variant model should process new options
 *
 * @param array $data The data being saved.
 * @param array $options The save options.
 *
 * @return bool Should allow the Variant to process new options
 */
	protected function _shouldProcessNewOptions($data, $options) {
		return (
			// Return true if there are options
			!empty($data['Options'])

			// Or there are no options and this is a new product being created
			|| empty($data[$this->alias][$this->primaryKey])

			// Or this product has already been created but the options have all been removed
			|| ($options['savingFromAdmin'] && empty($data['Options']))
		);
	}

/**
 * Before we save anything process the options into variants:
 * - also process the category array
 * - also process the variant images
 *
 * When a product is created:
 * - If options are provided then generate new variants,
 * - If variants are provided then save them against the product,
 * - If no options or variatns are provided then create a new variant that matched the product.
 *
 * When a product is edited:
 * - If CurrentOptions and Options are empty then no change has ocurred so don't add or delete variants,
 * - If CurrentOptions are provided and Options is empty then delete the variants and create a variant that matches
 * 	the product.
 * - If the CurrentOptions are empty and Options are provided empty then generate all new variants based on the options.
 * - If both CurrentOptions and Options are provided, check which variants need to be generated and which ones need to
 * 	be deleted.
 * - If variants are provided and the variant shouldn't be deleted then keep it. Otherwise don't save any that are
 * 	provided as they could be removed as their options are deselected.
 * - If Options haven't been provided and there is only 1 variant, update the variant's name with the product's name.
 *
 * @param array $data The data being saved.
 * @param array $options The save options.
 * @return array The modified data to be saved.
 */
	public function beforeBeforeSave($data, $options) {
		$Variant = EvClassRegistry::init('EvShop.Variant');

		if (empty($data['CurrentOptions'])) {
			$data['CurrentOptions'] = array();
		}
		if (empty($data['Options'])) {
			$data['Options'] = array();
		}

		//Options are the options that were selected when the product was submitted to be saved.
		$data['Options'] = array_map(
			function ($options) {
				return array_combine(
					$options,
					$options
				);
			},
			$data['Options']
		);

		//CurrentOptions are the options that were selected before the product was saved/edited.
		//Match the current options array with the selected options array
		$data['CurrentOptions'] = array_map(
			function ($options) {
				return array_combine(
					$options,
					$options
				);
			},
			$data['CurrentOptions']
		);

		// Take a copy of any current Variant data so it doesn't get overwritten later
		$originalVariants = array();
		if (isset($data['Variant'])) {
			$originalVariants = $data['Variant'];
		}

		/*
		 * When editing a product, work out which variants need deleting
		 * If the product hasn't been created yet then we can skip this process as there wouldn't be anything to
		 * delete.
		 */
		if (!empty($data[$this->alias][$this->primaryKey])) {
			$this->variantsToDelete = $Variant->processDeletedOptions(
				$data['CurrentOptions'],
				$data['Options'],
				$data[$this->alias][$this->primaryKey]
			);
		}

		// We need to process options for new products so that products without options get a variant,
		// ..as well as existing products if we have new options
		// ..as well as existing products if they removed all the options on an edit
		if ($this->_shouldProcessNewOptions($data, $options)) {
			$data['Variant'] = $Variant->processNewOptions(
				$data['CurrentOptions'],
				$data['Options'],
				$data[$this->alias]
			);

			//Unset the options as we don't want to save them against the product. The options are associated through
			//the variants.
			unset($data['Options']);
		}

		/*
		 * If we're updating a product with no options we need to keep the single variant's name in sync with the
		 * product name. We only want to perform this is there are no variants already being saved as this means
		 * that there were options selected.
		 */
		if (!empty($data[$this->alias][$this->primaryKey]) && empty($data['Variant'])) {
			$productVariants = $this->Variant->findAllByProductId($data[$this->alias][$this->primaryKey]);

			/*
			 * If there were more than one variants on this product then if they are all being removed a
			 * new variant will be created and the name will match the product.
			 */
			if (!empty($productVariants) && count($productVariants) <= 1) {
				$productVariant = array_shift($productVariants);

				$productVariant['Variant']['name'] = $data[$this->alias]['name'];

				$data['Variant'][] = $productVariant['Variant'];
			}
		}

		/*
		 * Merge the original into the list if it is not on the list to delete. This stops old variants that aren't
		 * being changed from being duplicated as variant save happens after product afterSave deletes the variants
		 * from variantsToDelete.
		 */
		$idsToDelete = !empty($this->variantsToDelete) ? array_keys($this->variantsToDelete) : [];
		if (!empty($originalVariants)) {
			foreach ($originalVariants as $variant) {
				if (!empty($variant['id']) && !in_array($variant['id'], $idsToDelete)) {
					$data['Variant'][] = $variant;
				}
			}
		}

		// process the categories
		$Category = EvClassRegistry::init('EvShop.Category');
		$this->categoriesToDelete = $Category->processDeletedCategories($data['CategoriesProduct']);

		// process the variant images
		if (isset($this->variantImageSlots)) {
			$VariantImage = EvClassRegistry::init('EvShop.VariantImage');
			$data = $VariantImage->processVariantImageOptions($data, $this->variantImageSlots);
		}

		return $data;
	}

	/**
	 * afterSave - delete any variants to be deleted
	 *
	 */
	public function afterSave($created, $options = array()) {
		parent::afterSave($created, $options);

		if (! empty($this->variantsToDelete)) {
			$Variant = EvClassRegistry::init('EvShop.Variant');

			$Variant->deleteAll(
				array(
					'Variant.id' => array_keys($this->variantsToDelete)
				)
			);
		}

		if (! empty($this->categoriesToDelete)) {
			$CategoriesProduct = EvClassRegistry::init('EvShop.CategoriesProduct');

			$CategoriesProduct->deleteAll(
				array(
					'CategoriesProduct.id' => $this->categoriesToDelete
				)
			);
		}

		// dispatch product saved event
		$this->getEventManager()->dispatch(
			new CakeEvent('EvShop.Model.Product.saved', $this, array(
				'created' => $created,
				'product' => $this->data['Product']
			))
		);

		$this->clearCache($this->id);
	}

	/**
	 * afterFind - Calculate tax prices for each variant
	 *
	 */

	public function afterFind($results, $primary = false) {
		if (!empty($results)) {
			if (CakePlugin::loaded('EvTax')) {
				$results = $this->calculateTaxPrices($results);
			}
		}

		return parent::afterFind($results, $primary);
	}

	/**
	 * Called after every deletion operation.
	 *
	 * @return void
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#afterdelete
	 */
	public function afterDelete() {
		$this->CategoriesProduct->deleteAll(['CategoriesProduct.product_id' => $this->id], false);
	}

	/**
	 * readForEdit - bring out the variants and calculate the options
	 *
	 */
	public function readForEdit($id, $query = array()) {
		$query['contain']['CategoriesProduct'] = array('Category');

		if (!empty(Configure::read('EvShop.showAttributes'))) {
			$query['contain']['ProductAttribute'] = [];
		}

		// get product and product category data
		$data = parent::readForEdit($id, $query);

		if (! empty($data['Product'])) {
			// re-align the category products array
			$data['CategoriesProduct'] = Hash::combine(
				$data['CategoriesProduct'],
				'{n}.category_id',
				'{n}'
			);

			// get variants out for the product, containing pricing, ready for
			// in-page editing through the admin
			$data['Variant'] = $this->Variant->readForManage($id);

			$data['Options'] = [];

			// assign array of active options to auto-check items under the Options tab
			if (! empty($data['Variant'])) {
				$data['Options'] = Hash::combine(
					$data['Variant'],
					'{n}.Option.{n}.id',
					'{n}.Option.{n}.id',
					'{n}.Option.{n}.option_group_id'
				);
			}
		}

		return $data;
	}

/**
 * readForView - Do the Variants separately so the callbacks will be run
 *
 * @param int|string $id ID of row to edit or a string value (such as a slug) that will be matched against whatever
 * field is passed in `$query['key']` (if present).
 *
 * @param array $params The db query array - can be used to pass in additional parameters such as contain
 *
 * @return array
 */
	public function readForView($id, $query = array()) {
		$query['contain']['CategoriesProduct'] = array('Category');

		if (Configure::read('EvShop.showBrands')) {
			$query['contain'][] = 'Brand';
		}

		$data = parent::readForView($id, $query);

		if (empty($data)) {
			return false;
		}

		// perform this on it's own rather then contain so the inventory behaviours / cakllbacks can be triggered
		$data['Variant'] = $this->Variant->readForProductView($data[$this->alias][$this->primaryKey]);

		if (! empty($data['Variant'])) {
			$data['Options'] = Hash::combine($data['Variant'], '{n}.Option.{n}.id', '{n}.Option.{n}.id', '{n}.Option.{n}.option_group_id');
		}

		if (! empty($data['CategoriesProduct'])) {
			$data['CategoriesProduct'] = Hash::combine($data['CategoriesProduct'], '{n}.category_id', '{n}');
		}

		$data = $this->defaultMetaDescription($data);

		return $data;
	}

	/**
	 * unless the meta description has been overridden
	 * set seo product meta description
	 *
	 * @param 	array 	$data 	Array of product data
	 * @return 	array 	$data 	Updated product data array with updated meta description
	 */
	public function defaultMetaDescription($data) {
		// If including VAT by default choose the price with VAT
		if (Configure::read('EvShop.displayTaxAsVat')) {
			$prices = Hash::extract($data, 'Variant.{n}.VariantPricing.{n}[currency_id=' . CakeSession::read('EvCurrency.currencyId') . '][price_incTax>0].price_incTax');
		} else {
			$prices = Hash::extract($data, 'Variant.{n}.VariantPricing.{n}[currency_id=' . CakeSession::read('EvCurrency.currencyId') . '][price>0].price');
		}

		if (!empty($prices)) {
			$currencies = Configure::read('currencies');
			sort($prices);
			$price = array_shift($prices);
			$price = CakeNumber::currency(
				$price,
				$currencies[CakeSession::read('EvCurrency.currencyId')]
			);

			// Merge price into the data array so it can be found using tokens
			$data = Hash::merge($data, ['Product' => ['price' => $price]]);
		}

		if (!empty($data['MetaData']['description'])) {
			$data['MetaData']['description'] = $this->_replaceTokens($data['MetaData']['description'], $data);
		} else {
			$data['MetaData']['description'] = $this->_getDefaultMetaDescription($data);
		}

		if (!empty($data['MetaData']['title'])) {
			$data['MetaData']['title'] = $this->_replaceTokens($data['MetaData']['title'], $data);
		}

		return $data;
	}

/**
 * An overridable function to set the default string
 *
 * @param bool $data The product data
 * @return string The default meta description
 */
	protected function _getDefaultMetaDescription($data) {
		if (!empty($data[$this->alias]['price'])) {
			$siteTitle = Configure::read('SiteSetting.general.site_title');
			$hasMultiple = (!empty($data['Variant']) && count($data['Variant']) > 1);
			$description = 'Order {' . $this->alias . '.name} ' . ($hasMultiple ? 'from' : 'for') . ' only {' . $this->alias . '.price} from ' . $siteTitle . ' now!';
			return $this->_replaceTokens($description, $data);
		}

		return '';
	}

/**
 * Replaces {tokens} for those found by keys in the array.
 * Paths can be used e.g. {Product.name} to get $data['Product']['name']
 *
 * @param string $subject The string to parse
 * @param array  $data    The product data
 * @return string The subject with the tokens replaced
 */
	protected function _replaceTokens($subject, $data) {
		$replaced = preg_replace_callback('/\{(.*)\}/U', function ($matches) use ($data) {
			$value = Hash::get($data, $matches[1]);
			if (!empty($value)) {
				return $value;
			}

			return '';
		}, $subject);
		return $replaced;
	}

	/**
	 * Calculate the prices with tax for each variant
	 *
	 * @param array $result The results from the find
	 * @return array $result Updated variant pricing to include tax prices
	 */

	public function calculateTaxPrices($result) {
		$TaxLevelModel = EvClassRegistry::init('EvTax.TaxLevel');
		$VariantPricingModel = EvClassRegistry::init('EvShop.VariantPricing');

		// Cache the tax levels in the model - should be safe in the same call.
		if (empty($this->_taxLevels)) {
			$this->_taxLevels = $TaxLevelModel->find('all', array(
				'conditions' => array(
					'is_active' => 1
				)
			));
			$this->_taxLevels = Hash::combine($this->_taxLevels, '{n}.TaxLevel.id', '{n}');
		}

		if (! empty($result)) {
			foreach ($result as $key => $item) {
				$taxLevel = array();

				if (isset($item['Product']['tax_level_id']) && $item['Product']['tax_level_id'] > 0) {

					$taxLevel = $this->_taxLevels[$item['Product']['tax_level_id']];

					// As well as the variants add tax to the product 'summary' prices
					if (isset($item['Product']['was_price_ex_vat'])) {
						if (isset($taxLevel['TaxLevel']['rate'])) {
							$result[$key]['Product']['was_price'] = $VariantPricingModel->addTaxToPrice($item['Product']['was_price_ex_vat'], $taxLevel[$TaxLevelModel->alias]['rate']);
						} else {
							$result[$key]['Product']['was_price'] = $item['Product']['was_price_ex_vat'];
						}
					}
					if (isset($item['Product']['rrp_price_ex_vat'])) {
						if (isset($taxLevel['TaxLevel']['rate'])) {
							$result[$key]['Product']['rrp_price'] = $VariantPricingModel->addTaxToPrice($item['Product']['rrp_price_ex_vat'], $taxLevel[$TaxLevelModel->alias]['rate']);
						} else {
							$result[$key]['Product']['rrp_price'] = $item['Product']['rrp_price_ex_vat'];
						}
					}

					if (isset($item['Variant']) && ! empty($item['Variant'])) {
						foreach ($item['Variant'] as $vKey => $variant) {

							if (! empty($taxLevel['TaxLevel'])) {

								if (isset($variant['VariantPricing']) && ! empty($variant['VariantPricing'])) {

									foreach ($variant['VariantPricing'] as $pricingKey => $pricing) {
										$variant['VariantPricing'][$pricingKey]['TaxLevel'] = $taxLevel['TaxLevel'];
									}
								}
							}

							if (isset($result[$key]['Variant'][$vKey])) {
								$result[$key]['Variant'][$vKey] = $VariantPricingModel->calculateTax($variant, 'Variant');
							}
						}
					}
				}
			}
		}

		return $result;
	}

/**
 * Clears the cache stored against this product
 *
 * @param int $id The product ID. If null, all of the products will be cleared.
 * @param string $key The cache key to delete. If null, all of the cached data against this product will be deleted
 * @return void
 */
	public function clearCache($id = null, $key = null) {
		$cacheName = Configure::read('EvShop.productCacheName');
		if (!empty($cacheName)) {
			if (!empty($id)) {
				if (empty($key)) {
					// Delete all product specific data
					Cache::delete('EvShop.Product.' . $id, $cacheName);
				} else {
					$cachedData = Cache::read('EvShop.Product.' . $id);
					unset($cachedData[$key]);
					if (empty($cachedData)) {
						// If it's empty delete the whole thing
						$this->clearCache($id);
					} else {
						// Else save the rest back to the cache
						Cache::write('EvShop.Product.' . $id, $cachedData, $cacheName);
					}
				}
			} else {
				Cache::clear(false, $cacheName);
			}
		}
	}

/**
 * Cache an item against a product
 *
 * @param int $id The product id to cache against
 * @param string $key The cache key
 * @param mixed $data The data to store. Must be serialisable
 * @return void
 */
	public function cache($id, $key, $data) {
		$cacheName = Configure::read('EvShop.productCacheName');
		if (!empty($cacheName)) {
			$cachedData = Cache::read('EvShop.Product.' . $id, $cacheName);
			$cachedData[$key] = $data;
			Cache::write('EvShop.Product.' . $id, $cachedData, $cacheName);
		}
	}

/**
 * Fetch cached data from a product
 *
 * @param int $id The product id to cache against
 * @param string $key The cache key. If null it will fetch the entire data stored against this product
 * @param mixed $data The data to store. Must be serialisable
 * @return void
 */
	public function fetchFromCache($id, $key = null) {
		$cacheName = Configure::read('EvShop.productCacheName');
		if (!empty($cacheName)) {
			$cachedData = Cache::read('EvShop.Product.' . $id, $cacheName);
			if (!empty($key)) {
				return isset($cachedData[$key]) ? $cachedData[$key] : null;
			}
			return $cachedData;
		}
		return null;
	}
}
