<?php
App::uses('EvShopAppModel', 'EvShop.Model');
/**
 * Option Model
 *
 * @property OptionGroup $OptionGroup
 * @property Variant $Variant
 */
class Option extends EvShopAppModel {

/**
 * Display field
 *
 * @var string
 */
	public $displayField = 'name';

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'option_group_id' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'Please select an option group'
			),
		),
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'An option name must be entered'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'OptionGroup' => array(
			'className' => 'EvShop.OptionGroup'
		)
	);

/**
 * hasAndBelongsToMany associations
 *
 * @var array
 */
	public $hasAndBelongsToMany = array(
		'Variant' => array(
			'className' => 'EvShop.Variant',
			'joinTable' => 'ev_shop_options_variants'
		),
		'CategoryOverride' => array(
			'className' => 'EvShop.CategoryOverride',
			'with' => 'EvShop.CategoryOverridesOption',
			'unique' => 'keepExisting',
		),
	);

/**
 * Constructor. Binds the model's database table to the object.
 *
 * @param bool|int|string|array $id Set this ID for this model on startup,
 * can also be an array of options, see above.
 * @param string $table Name of database table to use.
 * @param string $ds DataSource connection name.
 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (! empty(Configure::read('EvShop.manageOptions')) && !empty(Configure::read('EvShop.managedOptionsHaveImages'))) {
			// update the model image slots
			$this->imageSlots = [
				'main' => [
					'fields' => false,
					'slots' => 1
				]
			];
		}

		parent::__construct($id, $table, $ds);
	}

/**
 * Get the given options name data.
 *
 * @param array $options Array of option IDs.
 * @return array Array list of all options in id => name format
 */
	public function getOptionNames($options) {
		return $this->find(
			'list',
			array(
				'conditions' => array(
					'Option.id' => $options
				),
				'order' => [
					'Option.option_group_id ASC',
					'Option.sequence ASC',
					'Option.name ASC',
				],
				'callbacks' => false
			)
		);
	}

/**
 * Called before every deletion operation.
 *
 * @param bool $cascade If true records that depend on this record will also be deleted
 * @return bool True if the operation should continue, false if it should abort
 * @link https://book.cakephp.org/2.0/en/models/callback-methods.html#beforedelete
 */
	public function beforeDelete($cascade = true) {
		$beforeDelete = parent::beforeDelete($cascade);

		if ($beforeDelete !== true) {
			return $beforeDelete;
		}

		return !$this->isUsed($this->id);
	}

/**
 * Clear the product cache
 *
 * @param bool $created If a new entry was created
 * @param array $options The options used when saving
 * @return void
 */
	public function afterSave($created, $options = array()) {
		$this->Variant->Product->clearCache();
		parent::afterSave($created, $options);
	}

/**
 * Check if an option is in use on a product.
 *
 * @param int $id The id of the option to check.
 * @return bool.
 */
	public function isUsed($id) {
		$variantPrimaryKeyField = $this->Variant->alias . '.' . $this->Variant->primaryKey;

		return !empty(
			$this->Variant->find(
				'first',
				[
					'fields' => [
						$variantPrimaryKeyField,
					],
					'joins' => [
						[
							'table' => 'ev_shop_options_variants',
							'alias' => 'OptionsVariant',
							'conditions' => [
								'OptionsVariant.variant_id = ' . $variantPrimaryKeyField,
								'OptionsVariant.option_id' => $id,
							],
						]
					],
				]
			)
		);
	}
}
