<?php

App::uses('EvShopAppModel', 'EvShop.Model');

class CategoryOverride extends EvShopAppModel {

	public $belongsTo = [
		'Category' => [
			'className' => 'EvShop.Category',
		],
		'Brand' => [
			'className' => 'EvShop.Brand',
		],
	];

	public $hasAndBelongsToMany = [
		'Option' => [
			'className' => 'EvShop.Option',
			'with' => 'EvShop.CategoryOverridesOption',
			'unique' => 'keepExisting',
		],
		'ProductAttribute' => [
			'className' => 'EvShop.ProductAttribute',
			'with' => 'EvShop.CategoryOverridesProductAttribute',
			'unique' => 'keepExisting',
		],
	];

	public $validate = [
		'category_id' => [
			'required' => [
				'rule' => 'notBlank',
				'message' => 'Select a category for the category override',
			],
		],
		'pagination_page' => [
			'numeric' => [
				'rule' => 'numeric',
				'message' => 'Please provide a valid pagination page number',
				'allowEmpty' => true,
			],
			'validPageNumber' => [
				'rule' => ['comparison', '>=', 1],
				'message' => 'Please provide a valid pagination page number',
				'allowEmpty' => true,
			],
		],
	];

/**
 * Format the options into HABTM data.
 *
 * @param array $data The data being saved.
 * @param array $options The save options.
 * @return array The modified data to be saved.
 */
	public function beforeBeforeSave($data, $options) {
		if (!empty($data['Option']) && is_array($data['Option'])) {
			$options = $data['Option'];

			//Clear out the data because if it isn't formatted we don't want to save it.
			unset($data['Option']);

			$selectedOptions = [];

			foreach ($options as $optionGroupId => $optionId) {
				//Select the optionId in the group
				if (!empty($optionId)) {
					$selectedOptions[] = $optionId;
				}
			}

			if (!empty($selectedOptions)) {
				$data['Option']['Option'] = $selectedOptions;
			} else {
				$data['Option']['Option'] = [];
			}
		}

		if (!empty($data['ProductAttribute']) && is_array($data['ProductAttribute'])) {
			$productAttributes = $data['ProductAttribute'];

			//Clear out the data because if it isn't formatted we don't want to save it.
			unset($data['ProductAttribute']);

			$selectedOptions = [];

			foreach ($productAttributes as $productAttributeGroupId => $productAttributeId) {
				//Select the productAttributeId in the group
				if (!empty($productAttributeId)) {
					$selectedOptions[] = $productAttributeId;
				}
			}

			if (!empty($selectedOptions)) {
				$data['ProductAttribute']['ProductAttribute'] = $selectedOptions;
			} else {
				$data['ProductAttribute']['ProductAttribute'] = [];
			}
		}

		$overrideHash = $this->generateOverrideHash($this->_getHashSaveData($data));

		if (!empty($overrideHash)) {
			$data[$this->alias]['override_hash'] = $overrideHash;
		}

		return $data;
	}

/**
 * Query used to retrieve a record ready for edit.
 *
 * @param int   $id ID of row to edit.
 * @param array $params The db query array - can be used to pass in additional parameters such as contain.
 * @return array.
 */
	public function readForEdit($id, $params = []) {
		$data = parent::readForEdit($id, $params);

		if (!empty($data[$this->alias]['id'])) {
			//Find and format the override options
			$options = $this->Option->find(
				'list',
				[
					'joins' => [
						[
							'table' => 'ev_shop_category_overrides_options',
							'alias' => 'CategoryOverridesOption',
							'conditions' => [
								'CategoryOverridesOption.option_id = ' . $this->Option->alias . '.id',
								'CategoryOverridesOption.category_override_id' => $id,
							],
						]
					],
					'fields' => [
						$this->Option->alias . '.option_group_id',
						$this->Option->alias . '.id',
					],
				]
			);

			$data[$this->Option->alias] = $options;

			//Find and format the override product attributes
			$productAttributes = $this->ProductAttribute->find(
				'list',
				[
					'joins' => [
						[
							'table' => 'ev_shop_category_overrides_product_attributes',
							'alias' => 'CategoryOverridesProductAttribute',
							'conditions' => [
								'CategoryOverridesProductAttribute.product_attribute_id = ' . $this->ProductAttribute->alias . '.id',
								'CategoryOverridesProductAttribute.category_override_id' => $id,
							],
						]
					],
					'fields' => [
						$this->ProductAttribute->alias . '.product_attribute_group_id',
						$this->ProductAttribute->alias . '.id',
					],
				]
			);

			$data[$this->ProductAttribute->alias] = $productAttributes;
		}

		return $data;
	}

/**
 * Find an override based on the categoryId and a list of options. For an override to be found the list of options must
 * exactly match the list of options on the category override.
 *
 * @param array $hashData The data to hash and find an override with.
 * @param array $params   Custom query parameters.
 * @return array The found category override.
 */
	public function findOverride($hashData, $params = []) {
		$overrideHash = $this->generateOverrideHash($hashData);

		if (empty($overrideHash)) {
			return [];
		}

		$defaultParams = [
			'conditions' => [
				$this->alias . '.override_hash' => $overrideHash,
			],
			'contain' => [
				'Option',
				'ProductAttribute',
			],
		];

		$params = Hash::merge($defaultParams, $params);

		return $this->find('first', $params);
	}

/**
 * A convenience wrapper for findOverride which tells you if there's a an override is found.
 *
 * @param array $hashData The data to hash and find an override with.
 * @param array $params   Custom query parameters.
 * @return bool
 */
	public function hasOverride($hashData, $params = []) {
		return !empty($this->findOverride($hashData, $params));
	}

/**
 * Generate a hash for a category override using the category id, brand id and a list of options.
 *
 * @param array $hashData The data to hash.
 * @return null|string The generated hash, null if a hash could not be generated
 */
	public function generateOverrideHash($hashData) {
		$hashString = '';

		// Make sure the provided data is always in the same order.
		ksort($hashData);

		foreach ($hashData as $key => $value) {
			if (empty($value)) {
				continue;
			}

			/*
			 * Append the key and value in case some values are skipped but are equivalent, for example without the
			 * key, if options were passed in and no attributes were then the hash string would be "1,2,3;". With no
			 * options and some attributes being passed making the string "1,2,3;". These would be hashed to the
			 * same value but represent a different override, whereas with the key the strings would be
			 * "options:1,2,3;" and "attributes:1,2,3" so would get different hashes.
			 */
			if (!is_array($value)) {
				$hashString .= $key . ':' . $value . ';';
			} else {
				// Sort arrays of ids so that they are always in the same order.
				sort($value);

				$hashString .= $key . ':' . implode(',', $value) . ';';
			}
		}

		return md5($hashString);
	}

/**
 * Take the save data of a category override and return the hashable data.
 *
 * @param array $data Category override data.
 * @return array Data to hash.
 */
	protected function _getHashSaveData($data) {
		return [
			'categoryId' => !empty($data[$this->alias]['category_id']) ?
				$data[$this->alias]['category_id'] :
				null,
			'brandId' => !empty($data[$this->alias]['brand_id']) ?
				$data[$this->alias]['brand_id'] :
				null,
			'options' => !empty($data['Option']['Option']) ?
				$data['Option']['Option'] :
				[],
			'attributes' => !empty($data['ProductAttribute']['ProductAttribute']) ?
				$data['ProductAttribute']['ProductAttribute'] :
				[],
		];
	}

/**
 * Regenerate hashes of category overrides. If an id is provided then only that category override has it's hash
 * regenerated.
 *
 * @param int   $id     An id of a specific override to regenerate. Defaults to null so all overrides get regenerated.
 * @param array $params Custom query params to be used when regenerating specific overrides.
 * @return array An array containing the amount of overrides, amount that regenerated successfully and the amount that
 *               failed.
 */
	public function regenerateHashes($id = null, $params = []) {
		if (!empty($id)) {
			$params['conditions'][$this->alias]['id'] = $id;
		}

		$categoryOverrides = $this->find('list', $params);

		$regeneration = [
			'overrides' => count($categoryOverrides),
			'regenerated' => 0,
			'failed' => 0,
		];

		if (empty($categoryOverrides)) {
			return $regeneration;
		}

		foreach ($categoryOverrides as $overrideId => $override) {
			$override = $this->readForEdit($overrideId);

			$this->clear();
			$saved = $this->saveAssociated($override, ['deep' => true]);

			if ($saved) {
				$regeneration['regenerated']++;
			} else {
				$regeneration['failed']++;
			}
		}

		return $regeneration;
	}
}
