<?php
App::uses('EvShopAppModel', 'EvShop.Model');
/**
 * Category Model
 *
 * @property Product $Product
 */
class Category extends EvShopAppModel {

	public $actsAs = array(
		'Tree' => array(
			'level' => 'level',
		),
		'Routable.Routable' => array(
			'config' => 'EvShop',
			'alias' => 'category/:displayField/*'
		),
		'MetaData.Meta',
		'EvTemplates.Template' => array(
			'formInject' => true,
			'restrictTo' => 'Category'
		)
	);

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	public $order = array('Category.lft' => 'ASC');

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'A category name must be entered.'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be remove

	/**
	 * hasMany relationships
	 */
	public $hasMany = array(
		'ChildCategory' => array(
			'className' => 'EvShop.Category',
			'foreignKey' => 'parent_id'
		),
		'CategoriesProduct' => array(
			'className' => 'EvShop.CategoriesProduct',
			'order' => 'CategoriesProduct.sequence ASC'
		),
		'CategoryOverride' => array(
			'className' => 'EvShop.CategoryOverride',
		),
	);

	/**
	 * belongsTo relationships
	 */
	public $belongsTo = array(
		'ParentCategory' => array(
			'className' => 'EvShop.Category',
			'foreignKey' => 'parent_id'
		)
	);

	/**
	 * image slots
	 */
	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => false
		)
	);

	/**
	 * beforeSave / afterFind callbacks to transform parent_id
	 * into top level category option iuf selected
	 */
	/**
	 * Called after each find operation. Can be used to modify any results returned by find().
	 * Return value should be the (modified) results.
	 *
	 * @param mixed $results The results of the find operation
	 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
	 * @return mixed Result of the find operation
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#afterfind
	 */
	public function afterFind($results, $primary = false) {
		foreach ($results as $key => $item) {
			if (empty($item[$this->alias]['parent_id'])) {
				$results[$key][$this->alias]['parent_id'] = 0;
			}
		}

		return $results;
	}

	/**
	 * Called before each save operation, after validation. Return a non-true result
	 * to halt the save.
	 *
	 * @param array $options Options passed from Model::save().
	 * @return bool True if the operation should continue, false if it should abort
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#beforesave
	 * @see Model::save()
	 */
	public function beforeSave($options = array()) {
		$return = parent::beforeSave($options);

		if (empty($this->data[$this->alias]['parent_id'])) {
			$this->data[$this->alias]['parent_id'] = null;
		}

		return $return;
	}

	/**
	 * Called after every deletion operation.
	 *
	 * @return void
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#afterdelete
	 */
	public function afterDelete() {
		$this->CategoriesProduct->deleteAll(['CategoriesProduct.category_id' => $this->id], false);
	}

	/**
	 * check the selected categories to see which ones we need to unlink
	 *
	 * @param 	array 	The array of selected categories passed by reference
	 * @return 	array 	Array fo linker table rows to deleted
	 */
	public function processDeletedCategories(&$data) {
		$toDelete = array();

		if (!empty($data)) {
			foreach ($data as $categoryId => $category) {
				if (isset($category['id']) && ! isset($category['category_id'])) {

					$toDelete[] = $category['id']; // id of the linker table row
					unset($data[$categoryId]);
				}
			}
		}

		return $toDelete;
	}

	/**
	 * get the category list for use in calculating breadcrumb
	 *
	 * @return array
	 */
	public function getBreadcrumbList() {
		$categories = $this->find(
			'all',
			array(
				'conditions' => array(
					'Category.is_active' => 1
				),
				'callbacks' => false
			)
		);

		return Hash::combine(
			$categories,
			'{n}.Category.id',
			'{n}'
		);
	}

/**
 * Find categories by an associated product.
 * Only active categories that are directly associated with the product will be returned.
 *
 * @param array $productId The id of the associated product.
 * @param array $params    Additional query parameters.
 * @return array.
 */
	public function findByAssociatedProduct($productId, $params = []) {
		$params = $this->mergeQueryParams(
			[
				'joins' => [
					[
						'table' => 'ev_shop_categories_products',
						'alias' => 'CategoriesProduct',
						'conditions' => [
							'CategoriesProduct.category_id = ' . $this->alias . '.' . $this->primaryKey,
							'CategoriesProduct.product_id' => $productId,
						],
					]
				],
				'conditions' => [
					$this->alias . '.is_active' => true,
				],
			],
			$params
		);

		return $this->find('all', $params);
	}

/**
 * Find the parents of a category. Parents are found using the left and right tree values of a category.
 * Categories are returned in ascending level order, this means that the top most level parent will be the first
 * category to be returned, followed by it's children.
 * The provided category is not included in the results.
 *
 * @param array $category Category data.
 * @param array $params   Additional query parameters.
 * @return array.
 */
	public function findParents($category, $params = []) {
		if (isset($category[$this->alias])) {
			$category = $category[$this->alias];
		}

		if (empty($category['lft']) || empty($category['rght'])) {
			return [];
		}

		$params = $this->mergeQueryParams(
			[
				'conditions' => [
					$this->alias . '.lft <' => $category['lft'],
					$this->alias . '.rght >' => $category['rght'],
				],
				'order' => [
					$this->alias . '.level' => 'ASC',
				],
			],
			$params
		);

		return $this->find('all', $params);
	}
}
